#pragma once

#include "debug/log.hpp"
#include "fundamentals/helpers.hpp"

#include <pajlada/settings/settingmanager.hpp>

#include <cstdint>
#include <functional>
#include <memory>
#include <mutex>
#include <vector>

namespace Vape {

template <typename InnerType>
class ObjectManager
{
public:
    using ObjectConstructor = std::function<InnerType(uint32_t)>;
    using OnLoadFunction    = std::function<void(InnerType &)>;

    ObjectManager(const std::string &_objectRootPath, ObjectConstructor _objectConstructor,
                  OnLoadFunction _onLoad)
        : objectRootPath(_objectRootPath)
        , objectConstructor(std::move(_objectConstructor))
        , onLoad(std::move(_onLoad))
        , objectIndicesMutex(new std::mutex)
    {
    }

    ObjectManager(ObjectManager &&other)
        : objectRootPath(std::move(other.objectRootPath))
        , objectConstructor(std::move(other.objectConstructor))
        , onLoad(std::move(other.onLoad))
        , objects(std::move(other.objects))
        , objectIndices(std::move(other.objectIndices))
        , objectIndicesMutex(std::move(other.objectIndicesMutex))
    {
    }

    void
    LoadObjects()
    {
        // Check how many objects are saved
        auto numObjects = pajlada::Settings::SettingManager::arraySize(this->objectRootPath);

        // Loop through each index of the /objects array
        for (uint32_t objectIndex = 0; objectIndex < numObjects; ++objectIndex) {
            if (pajlada::Settings::SettingManager::isNull(
                    fS("{}/{}", this->objectRootPath, objectIndex))) {
                // The array value at index `objectIndex` is null, skip this entry
                continue;
            }

            // Set index `objectIndex` as used in `objectIndices` vector
            // Create Object with `objectIndex`
            // Create right-click "Remove" option for Object in browser
            // Call onLoad with object as a reference
            // Place Object into `objects` vector
            this->LoadObject(objectIndex);
        }
    }

    // Can return nullptr
    rapidjson::Value *
    GetJSON(bool save = true)
    {
        if (save) {
            pajlada::Settings::SettingManager::save();
        }

        return pajlada::Settings::SettingManager::rawValue(this->objectRootPath.c_str());
    }

    void
    LoadObject(uint32_t objectIndex)
    {
        InnerType object = this->CreateObject(objectIndex);

        this->onLoad(object);

        this->objects.emplace_back(std::move(object));
    }

private:
    // Create object with the given index
    InnerType
    CreateObject(uint32_t objectIndex)
    {
        {
            std::lock_guard<std::mutex> lock(*this->objectIndicesMutex);

            this->objectIndices.push_back(objectIndex);
        }

        return this->objectConstructor(objectIndex);
    }

    const std::string objectRootPath;
    ObjectConstructor objectConstructor;
    OnLoadFunction onLoad;

protected:
    std::vector<InnerType> objects;

    std::vector<uint32_t> objectIndices;
    mutable std::unique_ptr<std::mutex> objectIndicesMutex;
};

}  // namespace Vape
