#pragma once

#include <pajlada/signals/signal.hpp>

#include <map>
#include <mutex>
#include <set>
#include <vector>

namespace Vape {

template <typename ObjectType>
class ObjectStore
{
public:
    static ObjectStore &
    instance()
    {
        static ObjectStore store;

        return store;
    }

    void
    doRegister(ObjectType *object)
    {
        std::lock_guard<std::mutex> lock(this->objectMutex);

        // TODO(pajlada): Ensure that the given path is unused
        auto &set = this->objectMap[object->path];
        set.insert(object);
    }

    void
    doUnregister(ObjectType *object)
    {
        std::lock_guard<std::mutex> lock(this->objectMutex);

        // TODO(pajlada): Ensure that the object at the path is `object`
        auto &set = this->objectMap[object->path];
        set.erase(object);
    }

    const std::vector<ObjectType *>
    getObjects() const
    {
        std::lock_guard<std::mutex> lock(this->objectMutex);

        std::vector<ObjectType *> ret;

        for (const auto &it : this->objectMap) {
            const auto &set  = it.second;
            auto firstObject = set.begin();

            if (firstObject != set.end()) {
                ret.push_back(*firstObject);
            }
        }

        return ret;
    }

    // Returns whatever pointer happens to be first in the given object set
    ObjectType *
    getObject(const std::string &path) const
    {
        std::lock_guard<std::mutex>(this->objectMutex);

        auto it = this->objectMap.find(path);

        if (it == this->objectMap.end()) {
            return nullptr;
        }

        const std::set<ObjectType *> &set = it->second;

        if (set.empty()) {
            return nullptr;
        }

        return *set.begin();
    }

    pajlada::Signals::Signal<ObjectType *> objectRegistered;
    pajlada::Signals::Signal<ObjectType *> objectUnregistered;

protected:
    std::map<std::string, std::set<ObjectType *>> objectMap;

    mutable std::mutex objectMutex;
};

}  // namespace Vape
