#include <boost/optional.hpp>

#include <deque>
#include <mutex>

namespace Vape {

// OutputQueue is specialized for senders that need to send data on a single thread
// Each pass when the send has been finished, it should check whether or not to call the send again
template <class T>
struct OutputQueue {
    // Push adds an entry at the end of the queue
    // Returns false if the queue was empty when we called Push
    // otherwise, returns true
    bool
    Push(T &&entry)
    {
        std::unique_lock<std::mutex> lock(this->mutex);

        this->data.emplace_back(std::move(entry));

        return this->data.size() > 1;
    }

    // Front returns the front-most value of the queue
    // If the queue is empty, this returns an empty optional
    boost::optional<T>
    Front()
    {
        std::unique_lock<std::mutex> lock(this->mutex);

        if (this->data.empty()) {
            return boost::none;
        }

        return this->data.front();
    }

    // Pop removes the front-most value in the queue
    // If the value removed was the only value in the queue, return true
    // Otherwise, return false
    bool
    Pop()
    {
        std::unique_lock<std::mutex> lock(this->mutex);

        assert(!this->data.empty());

        this->data.pop_front();

        return this->data.empty();
    }

    // Clear removes all values in the queue
    void
    Clear()
    {
        std::unique_lock<std::mutex> lock(this->mutex);

        this->data.clear();
    }

private:
    std::mutex mutex;
    std::deque<T> data;
};

}  // namespace Vape
