#pragma once

#include <pajlada/serialize.hpp>
#include <rapidjson/document.h>
#include <rapidjson/prettywriter.h>
#include <rapidjson/schema.h>
#include <rapidjson/stringbuffer.h>
#include <rapidjson/writer.h>

namespace Vape::rj {

void SetResponseMessageTarget(const rapidjson::Document &message, rapidjson::Document &response);

rapidjson::Document CreateResponse(const rapidjson::Document &root);
rapidjson::Document CreateTargetedResponse(const rapidjson::Document &root);

template <typename Encoding, typename Allocator>
rapidjson::GenericValue<Encoding, Allocator> &
AddMember(rapidjson::GenericDocument<Encoding, Allocator> &object, const char *key,
          rapidjson::GenericValue<Encoding, Allocator> value)
{
    auto &a = object.GetAllocator();

    object.AddMember(rapidjson::GenericValue<Encoding, Allocator>(key, a).Move(),  // Key
                     value,                                                        // value
                     a);

    return object[key];
}

template <typename Encoding, typename Allocator>
rapidjson::GenericValue<Encoding, Allocator> &
AddMember(rapidjson::GenericValue<Encoding, Allocator> &object, const char *key,
          typename rapidjson::GenericValue<Encoding, Allocator> value, Allocator &a)
{
    object.AddMember(rapidjson::GenericValue<Encoding, Allocator>(key, a).Move(),  // Key
                     value,                                                        // value
                     a);

    return object[key];
}

template <typename RJDocument>
typename RJDocument::ValueType &
AddMemberCopy(RJDocument &object, const char *key, typename RJDocument::ValueType &value)
{
    auto &a = object.GetAllocator();

    typename RJDocument::ValueType tmp(value, a);

    object.AddMember(rapidjson::Value(key, a).Move(),  // Key
                     tmp.Move(),                       // value
                     a);

    return object[key];
}

// Both the object and the value need to share the same lifetime options, hence the use of RJObject::ValueType instead of RJValue
template <typename RJObject, typename RJValue>
void
AddMemberCopy(RJObject &object, const char *key, const RJValue &value,
              typename RJObject::AllocatorType &a)
{
    // Make a copy of value using object's allocator (a)
    typename RJObject::ValueType tmp(value, a);

    // Move the copy into the object
    object.AddMember(typename RJObject::ValueType(key, a).Move(),  // Key
                     tmp.Move(),                                   // value
                     a);
}

template <typename Type, typename RJValue>
bool
Get(const RJValue &object, const rapidjson::GenericStringRef<char> &key, Type &in)
{
    if (!object.IsObject()) {
        return false;
    }

    auto memberIt = object.FindMember(key);
    if (memberIt == object.MemberEnd()) {
        return false;
    }

    bool error = false;

    auto v = pajlada::Deserialize<Type, typename RJValue::ValueType>::get(memberIt->value, &error);
    if (error) {
        return false;
    }

    std::swap(in, v);

    return true;
}

template <typename Type, typename RJValue>
bool
Get(const RJValue &object, const char *key, Type &in)
{
    auto keyRef = rapidjson::StringRef(key);
    return Get(object, keyRef, in);
}

template <typename Type, typename RJDocument>
void
Set(RJDocument &object, const char *key, const Type &value)
{
    assert(object.IsObject());

    AddMember(
        object, key,
        pajlada::Serialize<Type, typename RJDocument::ValueType>::get(value, object.GetAllocator()),
        object.GetAllocator());
}

template <typename RJValue, typename Type>
void
Set(RJValue &object, const char *key, const Type &value, typename RJValue::AllocatorType &a)
{
    assert(object.IsObject());

    AddMember(object, key, pajlada::Serialize<Type, RJValue>::get(value, a), a);
}

template <typename RJDocument, typename Type>
void
PushBack(RJDocument &array, const Type &value)
{
    auto &a = array.GetAllocator();

    assert(array.IsArray());

    array.PushBack(pajlada::Serialize<Type>::get(value, a), a);
}

template <typename RJValue, typename Type>
void
PushBack(RJValue &array, const Type &value, typename RJValue::AllocatorType &a)
{
    assert(array.IsArray());

    RJValue serializedValue = pajlada::Serialize<Type, RJValue>::get(value, a);

    array.PushBack(serializedValue, a);
}

template <typename RJValue>
void
PushBack(RJValue &array, RJValue &&value, typename RJValue::AllocatorType &a)
{
    assert(array.IsArray());

    array.PushBack(value, a);
}

template <typename Type>
void
GetIndex(const rapidjson::Value &array, const int index, Type &in)
{
    assert(array.IsArray());

    auto &val = array[index];
    //assert(val.Is<Type>());

    in = val.Get<Type>();
}

bool Parse(rapidjson::Document &document, const std::string &str);

template <typename Type>
rapidjson::Document
SerializeToDocument(const Type &val)
{
    rapidjson::Document doc;
    auto j = pajlada::Serialize<Type>::get(val, doc.GetAllocator());
    j.Swap(doc);

    return doc;
}

template <typename RJValue>
std::string
Stringify(const RJValue &v)
{
    rapidjson::StringBuffer buffer;
    rapidjson::Writer<rapidjson::StringBuffer> writer(buffer);
    v.Accept(writer);

    return std::string(buffer.GetString());
}

std::string StringifyPretty(const rapidjson::Value &v);

rapidjson::SchemaDocument MakeJSONSchema(const char *schemaJSON);

void PrintSchemaError(const std::string &h, rapidjson::SchemaValidator &validator);

}  // namespace Vape::rj
