#pragma once

#include "fundamentals/helpers.hpp"
#include "fundamentals/string-helpers.hpp"

#include <boost/any.hpp>
#include <boost/optional.hpp>
#include <rapidjson/document.h>

#include <map>
#include <string>

namespace Vape {

namespace detail {

static inline std::string
SettingsPath(const std::string &input)
{
    if (StartsWithChar(input, '@')) {
        return input.substr(1) + "_";
    }

    return input + "_";
}

}  // namespace detail

namespace Settings {

using UnbakedOptionsType = std::vector<std::pair<std::string, boost::any>>;

// Helper function for generating a parameter list in the following format:
// {
//   "options": {
//     listKey: listValue,
//   }
// }
inline std::map<std::string, boost::any>
Options(const UnbakedOptionsType &list)
{
    std::map<std::string, boost::any> parameters;

    parameters["options"] = JSMap(list, [](const auto &o) {
        std::map<std::string, boost::any> option;
        option["name"]  = o.first;
        option["value"] = o.second;
        return boost::any(option);
    });

    return parameters;
}

}  // namespace Settings

constexpr const char *DEFAULTS_PREFIX   = "/_defaults";
constexpr const char *PARAMETERS_PREFIX = "/_parameters";

struct ValueUpdateAction {
    std::string path;
    rapidjson::Value value;
    bool isTrigger;
};

struct ConstValueUpdateAction {
    ConstValueUpdateAction(std::string _path, const rapidjson::Value &_value, bool _isTrigger)
        : path(std::move(_path))
        , value(_value)
        , isTrigger(_isTrigger)
    {
    }

    std::string path;
    const rapidjson::Value &value;
    bool isTrigger;
};

boost::optional<ValueUpdateAction> ParseValueUpdateAction(rapidjson::Value &message);
boost::optional<ConstValueUpdateAction> ParseValueUpdateAction(const rapidjson::Value &message);

}  // namespace Vape
