#pragma once

#include "fundamentals/ipollable.hpp"
#include "signal-listener.hpp"

#include <memory>

namespace Vape::Signal {

/**
 * A group of signals that gets flushed together
 * When a indirect signal is called, it appends itself to the call list
 * Order of signals emitted from the same thread is maintained
 */
class Group : public IPollable
{
    using CallType     = std::pair<std::function<void()>, std::weak_ptr<AnyListener>>;
    using CallListType = moodycamel::ConcurrentQueue<CallType>;

    /**
     * A list of weak pointers to the listeners that has been called since last flush
     * in the order they were called
     */
    std::shared_ptr<CallListType> callList = std::make_shared<CallListType>();

public:
    /**
     * Get a shared pointer to the private list of calls
     */
    std::shared_ptr<CallListType> GetList();

    /**
     * Flush all listeners in this group
     * @return Number of callbacks flushed
     */
    size_t Flush();

    virtual void Poll() final;
};

inline std::shared_ptr<Group::CallListType>
Group::GetList()
{
    return this->callList;
}

inline size_t
Group::Flush()
{
    size_t i = 0;

    CallType l;

    while (this->callList->try_dequeue(l)) {
        /**
         *  Only invoke if the listener is still alive
         * Since Flush() is expected to be run from the same thread that manages the listener,
         * there should be no race condition where listener is destroyed during invokation
         */
        if (auto listener = l.second.lock()) {
            l.first();
            i++;
        }
    }

    return i;
}

inline void
Group::Poll()
{
    this->Flush();
}
}  // namespace Vape::Signal
