#pragma once
#include "concurrentqueue.h"

#include <functional>
#include <memory>
#include <tuple>
#include <vector>

namespace Vape::Signal {

/**
 * Forward declaration of Provider
 */
template <typename... Args>
class Provider;

/**
 * Base class of any listener type with the only purpose of being able to
 * manage lifetime of listeners without knowing its type
 */
class AnyListener
{
protected:
    virtual ~AnyListener() = default;
};

/**
 * Shared pointer to any kind of listener
 */
typedef std::shared_ptr<AnyListener> ListenerPtr;

/**
 * Vector of shared pointers to any kind of listener
 * Used to keep listeners alive when you don't care about their type after creation
 */
typedef std::vector<ListenerPtr> ListenerPtrList;

/**
 * Base class that all indirect listeners inherit from
 */
class SignalIndirectListenerBase : public AnyListener
{
protected:
    virtual ~SignalIndirectListenerBase() = default;
};

/**
 * Base type that all signal listeners inherit from
 */
template <typename... Args>
class Listener
{
protected:
    virtual ~Listener() = default;

    /**
     * Pure virtual: Invoke listener
     * @param args the templated parameters of the signal
     */
    virtual void Call(const Args &... args) = 0;

    friend class Provider<Args...>;
};

/**
 * A listener that fires immediately when called on the provider thread
 */
template <typename L, typename... Args>
class SignalDirectListener
    : public Listener<Args...>
    , public AnyListener
    , public std::enable_shared_from_this<SignalDirectListener<L, Args...>>
{
    L fn;

    /**
     * Invoke listener, inherited from Listener
     * Stores the call in the calls vector
     * @param args the templated parameters of the signal
     */
    void Call(const Args &... args) final;

public:
    /**
     * Construct a new direct listener
     * @param _fn The lambda that gets called when listener is invoked
     */
    SignalDirectListener(L _fn);
    ~SignalDirectListener() = default;

    /**
     * Factory function
     * @param fn The lambda that gets called when listener is invoked
     */
    static std::shared_ptr<SignalDirectListener<L, Args...>> Create(L fn);
};

template <typename L, typename... Args>
SignalDirectListener<L, Args...>::SignalDirectListener(L _fn)
    : fn(_fn)
{
}

template <typename L, typename... Args>
void
SignalDirectListener<L, Args...>::Call(const Args &... args)
{
    this->fn(args...);
}

template <typename L, typename... Args>
std::shared_ptr<SignalDirectListener<L, Args...>>
SignalDirectListener<L, Args...>::Create(L fn)
{
    return std::make_shared<SignalDirectListener<L, Args...>>(fn);
}

/**
 * A listener that stores calls and fires them in order when flushed
 */
template <typename L, typename... Args>
class SignalIndirectListener
    : public Listener<Args...>
    , public SignalIndirectListenerBase
    , public std::enable_shared_from_this<SignalIndirectListener<L, Args...>>
{
public:
    using CallType     = std::pair<std::function<void()>, std::weak_ptr<AnyListener>>;
    using CallListType = moodycamel::ConcurrentQueue<CallType>;

private:
    /**
     * Weak pointer to the groups call list
     */
    std::weak_ptr<CallListType> weakCallList;

    /**
     * The lambda that gets called when flushed
     */
    L fn;

    /**
     * Invoke listener, inherited from Listener
     * Stores the call in the calls vector
     * @param args the templated parameters of the signal
     */
    void Call(const Args &... args) final;

public:
    /**
     * Construct a new indirect listener
     * @param _weakCallList Weak pointer to the call list of the Group
     * @param _fn The lambda that gets called when listener is flushed
     */
    SignalIndirectListener(std::weak_ptr<CallListType> _weakCallList, L _fn);

    ~SignalIndirectListener() = default;

    /**
     * Factory function
     * @param weakCallList Weak pointer to the call list of the Group
     * @param fn The lambda that gets called when listener is flushed
     */
    static std::shared_ptr<SignalIndirectListener<L, Args...>> Create(
        std::weak_ptr<CallListType> weakCallList, L fn);
};

template <typename L, typename... Args>
SignalIndirectListener<L, Args...>::SignalIndirectListener(
    std::weak_ptr<CallListType> _weakCallList, L _fn)
    : weakCallList(_weakCallList)
    , fn(_fn)
{
}

template <typename L, typename... Args>
void
SignalIndirectListener<L, Args...>::Call(const Args &... args)
{
    // Capture args by value into lambda conforming to std::function<void()>
    auto cb = [this, args...]() { this->fn(args...); };

    if (auto lockedCallList = this->weakCallList.lock()) {
        // Enqueue call in group
        lockedCallList->enqueue({cb, this->shared_from_this()});
    }
}

template <typename L, typename... Args>
std::shared_ptr<SignalIndirectListener<L, Args...>>
SignalIndirectListener<L, Args...>::Create(std::weak_ptr<CallListType> weakCallList, L fn)
{
    return std::make_shared<SignalIndirectListener<L, Args...>>(weakCallList, fn);
}

}  // namespace Vape::Signal
