#pragma once
#include "signal-group.hpp"
#include "signal-listener.hpp"

#include <memory>
#include <vector>

namespace Vape::Signal {

template <typename... Args>
class Provider
{
    /**
     * Lock-free queue of new listeners
     */
    moodycamel::ConcurrentQueue<std::weak_ptr<Listener<Args...>>> newListeners;

    /**
     * List of listeners
     */
    std::vector<std::weak_ptr<Listener<Args...>>> listenerList;

    /**
     * Add listener to the list
     */
    std::shared_ptr<Listener<Args...>> AddListener(std::shared_ptr<Listener<Args...>> listener);

public:
    /**
     * Create a new direct listener and add it to the list
     * @return A shared pointer to the base listener class
     */
    template <typename L>
    std::shared_ptr<SignalDirectListener<L, Args...>> AddDirectListener(L fn);

    /**
     * Create a new indirect listener belonging to a listener group and add it to the list
     * @return A shared pointer to the base listener class
     */
    template <class Group, typename L>
    std::shared_ptr<SignalIndirectListener<L, Args...>> AddIndirectListener(Group &group, L fn);

    /**
     * Emit a signal to all listeners
     */
    void Emit(const Args &...);

    /**
     * Approximate listener count
     * Sum of added and queued listeners
     * Includes dead listeners not yet removed by Emit call
     */
    size_t ApproxListenerCount();
};

template <typename... Args>
std::shared_ptr<Listener<Args...>>
Provider<Args...>::AddListener(std::shared_ptr<Listener<Args...>> listener)
{
    this->newListeners.enqueue(listener);
    return listener;
}

template <typename... Args>
template <typename L>
std::shared_ptr<SignalDirectListener<L, Args...>>
Provider<Args...>::AddDirectListener(L fn)
{
    auto listener = SignalDirectListener<L, Args...>::Create(fn);
    this->AddListener(listener);
    return listener;
}

template <typename... Args>
template <class Group, typename L>
std::shared_ptr<SignalIndirectListener<L, Args...>>
Provider<Args...>::AddIndirectListener(Group &group, L fn)
{
    auto listener = SignalIndirectListener<L, Args...>::Create(group.GetList(), fn);
    this->AddListener(listener);
    return listener;
}

template <typename... Args>
void
Provider<Args...>::Emit(const Args &... t)
{
    {
        // Update list of listeners with additions since last Emit
        std::weak_ptr<Listener<Args...>> newListener;
        while (this->newListeners.try_dequeue(newListener)) {
            this->listenerList.emplace_back(std::move(newListener));
        }
    }

    auto fn = [&t...](auto &item) -> bool {
        if (auto lockedItem = item.lock()) {
            lockedItem->Call(t...);
            return false;
        }

        // Dead pointer
        return true;
    };

    // Iterate over listeners and remove dead weak_ptrs as we go
    this->listenerList.erase(
        std::remove_if(this->listenerList.begin(), this->listenerList.end(), fn),
        this->listenerList.end());
}

template <typename... Args>
size_t
Provider<Args...>::ApproxListenerCount()
{
    return this->listenerList.size() + this->newListeners.size_approx();
}

}  // namespace Vape::Signal
