#pragma once

#include <codecvt>
#include <limits>
#include <locale>
#include <string>
#include <vector>

namespace Vape {

inline std::wstring
s2ws(const std::string &str)
{
    using convert_typeX = std::codecvt_utf8<wchar_t>;
    std::wstring_convert<convert_typeX, wchar_t> converterX;

    return converterX.from_bytes(str);
}

inline std::string
ws2s(const std::wstring &wstr)
{
    using convert_typeX = std::codecvt_utf8<wchar_t>;
    std::wstring_convert<convert_typeX, wchar_t> converterX;

    return converterX.to_bytes(wstr);
}

template <typename Iterator>
inline std::string
JoinRange(Iterator begin, Iterator end, char separator)
{
    std::string result;

    if (begin != end) {
        result.append(*begin);
        ++begin;
    }

    for (; begin != end; ++begin) {
        result.push_back(separator);
        result.append(*begin);
    }

    return result;
}

// this is not safe with unicode strings. TODO: Replace with boost::locale
inline std::string
ToLower(std::string str)
{
    auto const toLower = [](char &c) {
        if (c >= 'A' && c <= 'Z') {
            c += 'a' - 'A';
        }

        return c;
    };

    for (char &c : str) {
        c = toLower(c);
    }

    return str;
}

// Split up the input string `str` into at-most `times` parts based on the given `delimiter`
inline std::vector<std::string>
Split(const std::string &str, const std::string &delimiter,
      std::size_t times = std::numeric_limits<std::size_t>::max())
{
    std::vector<std::string> parts;

    if (times == 0) {
        return parts;
    }

    std::size_t start = 0;
    auto end          = str.find(delimiter);

    while ((times-- > 0) && (end != std::string::npos)) {
        parts.emplace_back(str.substr(start, end - start));
        start = end + delimiter.length();
        end   = str.find(delimiter, start);
    }

    // Insert the remainder
    // TODO: Make sure we're not inserting an empty string at the end?
    auto remainder = str.substr(start, end);
    if (!remainder.empty()) {
        parts.emplace_back(str.substr(start, end));
    }

    return parts;
}

inline bool
StartsWith(const std::string &haystack, const std::string &needle)
{
    if (needle.size() > haystack.length()) {
        return false;
    }

    return haystack.compare(0, needle.size(), needle) == 0;
}

inline bool
StartsWithChar(const std::string &input, const char test)
{
    if (input.empty()) {
        return false;
    }

    return input[0] == test;
}

// TODO: Test
inline bool
EndsWith(const std::string &haystack, const std::string &needle)
{
    if (needle.size() > haystack.length()) {
        return false;
    }

    return haystack.compare(haystack.size() - needle.size(), haystack.size(), needle) == 0;
}

inline bool
EndsWithChar(const std::string &input, const char test)
{
    if (input.empty()) {
        return false;
    }

    return input[input.length() - 1] == test;
}

}  // namespace Vape
