#include <condition_variable>
#include <memory>
#include <mutex>

namespace Vape {
/**
 * Provides the functionality to wait for another thread to reach a certain sync point
 * Intended to be passed as an argument to a new thread
 */
class SyncPoint
{
    friend class WorkerToken;

    std::condition_variable cv;
    std::mutex m;
    bool done = false;

public:
    /**
     * A token grabbed by the worker thread that notifies parent thread on destruction
     */
    class WorkerToken
    {
        friend class SyncPoint;

        std::unique_lock<std::mutex> lock;
        SyncPoint& syncPoint;

    public:
        WorkerToken(SyncPoint &);
        ~WorkerToken();
    };

    std::unique_ptr<WorkerToken> GetWorkerToken();
    void WaitForDone();
};

std::unique_ptr<SyncPoint::WorkerToken> inline SyncPoint::GetWorkerToken()
{
    auto token = std::make_unique<WorkerToken>(*this);
    return token;
}

void inline SyncPoint::WaitForDone()
{
    std::unique_lock lock(this->m);
    this->cv.wait(lock, [this] { return this->done; });
}

inline SyncPoint::WorkerToken::WorkerToken(SyncPoint& syncPoint_)
    : lock(syncPoint_.m)
    , syncPoint(syncPoint_)
{
}

inline SyncPoint::WorkerToken::~WorkerToken()
{
    this->syncPoint.done = true;
    this->lock.unlock();
    this->syncPoint.cv.notify_one();
}

}  // namespace Vape
