#include "fundamentals/decklink.hpp"

#include "debug/log.hpp"

#include <algorithm>
#include <iostream>

using namespace std;

namespace Vape {

static bool decklink_initialized = false;

static string
DeckLinkGetOutputName(IDeckLink *deckLink)
{
    if (deckLink == nullptr) {
        return "";
    }

    string deckLinkName;
    STRINGOBJ internalName = nullptr;

    // Get model name in BSTR string
    auto result = deckLink->GetDisplayName(&internalName);
    if (result != S_OK) {
        return "";
    }

    // Convert internal name string to string
    StringToStdString(internalName, deckLinkName);
    STRINGFREE(internalName);

    return deckLinkName;
}

static bool
InitDecklinkAPI()
{
    if (decklink_initialized == true) {
        return true;
    }

    HRESULT result = Initialize();
    if (SUCCEEDED(result)) {
        decklink_initialized = true;
    } else {
        cerr << "Failed to initialize DeckLink API" << endl;
    }

    return decklink_initialized;
}

IDeckLinkOutput *
GetDeckLinkOutput(int index)
{
    const std::string h("GetDeckLinkOutput");

    if (!InitDecklinkAPI()) {
        Log::TError(h, "Unable to init DeckLink API");
        return nullptr;
    }

    IDeckLinkIterator *deckLinkIterator = nullptr;

    // Create an IDeckLinkIterator object to enumerate all DeckLink cards in the system
    HRESULT result = GetDeckLinkIterator(&deckLinkIterator);

    if (result != S_OK || deckLinkIterator == nullptr) {
        Log::TError(h, "Failed to get IDeckLinkIterator");
        return nullptr;
    }

    ScopedDeckLinkPointer<IDeckLinkIterator> scopedDeckLinkIterator(deckLinkIterator);

    // Iterate all cards in this system
    int i                           = 0;
    IDeckLink *deckLink             = nullptr;
    IDeckLinkOutput *deckLinkOutput = nullptr;
    while (deckLinkIterator->Next(&deckLink) == S_OK) {
        ScopedDeckLinkPointer<IDeckLink> scopedDeckLink(deckLink);

        if (i == index) {
            string deckLinkName;
            STRINGOBJ internalName = nullptr;

            // Get model name in BSTR string
            result = deckLink->GetDisplayName(&internalName);
            if (result != S_OK) {
                Log::TError(h, "Unable to get display name");
                return nullptr;
            }

            // Convert internal name string to string
            StringToStdString(internalName, deckLinkName);
            STRINGFREE(internalName);

            // Try to get output device, should fail if there is none for the card
            result = deckLink->QueryInterface(IID_IDeckLinkOutput,
                                              reinterpret_cast<void **>(&deckLinkOutput));
            if (result != S_OK || deckLinkOutput == nullptr) {
                Log::TError(h, "Found DeckLink with matchin index, but it has no output device: {}",
                            deckLinkName);
                return nullptr;
            }

            Log::TInfo(h, "Found DeckLink output: {}", deckLinkName);
            return deckLinkOutput;
        }

        ++i;
    }

    Log::TError(h, "Could not find DeckLink device matching index: {}", index);
    return nullptr;
}

IDeckLinkAttributes *
GetDeckLinkAttributes(IDeckLinkOutput *deckLinkOutput)
{
    IDeckLinkAttributes *attributes = nullptr;

    if (deckLinkOutput == nullptr) {
        Log::Error("GetDeckLinkAttributes called without valid IDeckLinkOutput");
        return nullptr;
    }

    auto result = deckLinkOutput->QueryInterface(IID_IDeckLinkAttributes,
                                                 reinterpret_cast<void **>(&attributes));
    if (result != S_OK) {
        Log::Error("QueryInterface failed for IID_IDeckLinkAttributes");
        return nullptr;
    }

    return attributes;
}

bool
CheckDeckLinkAttribute(IDeckLinkAttributes *attributes, BMDDeckLinkAttributeID attrID)
{
    if (!attributes) {
        Log::Error("CheckDeckLinkAttribute called without valid IDeckLinkAttributes");
        return false;
    }

    bool flagRes;
    auto result = attributes->GetFlag(attrID, &flagRes);
    if (result != S_OK) {
        return false;
    }

    return flagRes;
}

IDeckLinkConfiguration *
GetDeckLinkConfiguration(IDeckLinkOutput *deckLinkOutput)
{
    IDeckLinkConfiguration *configuration = nullptr;

    if (deckLinkOutput == nullptr) {
        Log::Error("GetDeckLinkConfiguration called without valid IDeckLinkOutput");

        return nullptr;
    }

    auto result = deckLinkOutput->QueryInterface(IID_IDeckLinkConfiguration,
                                                 reinterpret_cast<void **>(&configuration));
    if (result != S_OK) {
        Log::Error("QueryInterface failed for IID_IDeckLinkConfiguration");
        return nullptr;
    }

    return configuration;
}

bool
UpdateDeckLinkConfigurationFlag(IDeckLinkConfiguration *configuration,
                                BMDDeckLinkConfigurationID cfgID, bool val)
{
    if (configuration == nullptr) {
        Log::Error("UpdateDeckLinkConfigurationFlag called without IDeckLinkConfiguration");
        return false;
    }

    auto result = configuration->SetFlag(cfgID, val);
    if (result != S_OK) {
        Log::Error("Unable to set flag {}", cfgID);
        return false;
    }

    return true;
}

bool
UpdateDeckLinkConfigurationInt(IDeckLinkConfiguration *configuration,
                               BMDDeckLinkConfigurationID cfgID, int val)
{
    if (configuration == nullptr) {
        Log::Error("UpdateDeckLinkConfigurationInt called without IDeckLinkConfiguration");
        return false;
    }

    auto result = configuration->SetInt(cfgID, val);
    if (result != S_OK) {
        Log::Error("Unable to set int {}", cfgID);
        return false;
    }

    return true;
}

void
OnAllDeckLinkOutputs(const std::function<void(IDeckLinkOutput *, std::string)> &cb)
{
    if (!InitDecklinkAPI()) {
        return;
    }

    IDeckLinkIterator *deckLinkIterator = nullptr;

    // Create an IDeckLinkIterator object to enumerate all DeckLink cards in the system
    auto result = GetDeckLinkIterator(&deckLinkIterator);

    if (result != S_OK || deckLinkIterator == nullptr) {
        Log::Error("Failed to get IDeckLinkIterator");
        return;
    }

    ScopedDeckLinkPointer<IDeckLinkIterator> scopedDeckLinkIterator(deckLinkIterator);

    // Iterate all cards in this system
    IDeckLink *deckLink             = nullptr;
    IDeckLinkOutput *deckLinkOutput = nullptr;
    while (deckLinkIterator->Next(&deckLink) == S_OK) {
        ScopedDeckLinkPointer<IDeckLink> scopedDeckLink(deckLink);

        std::string deckLinkName;
        STRINGOBJ internalName = nullptr;

        // Get model name in BSTR string
        auto result = deckLink->GetDisplayName(&internalName);
        if (result != S_OK) {
            Log::Error("Unable to get decklink name of decklink device");
            continue;
        }

        // Convert internal name string to string
        StringToStdString(internalName, deckLinkName);
        STRINGFREE(internalName);

        // Try to get output device, should fail if there is none for the card
        result = deckLink->QueryInterface(IID_IDeckLinkOutput,
                                          reinterpret_cast<void **>(&deckLinkOutput));
        if (result != S_OK || deckLinkOutput == nullptr) {
            Log::Error("Unable to get decklink output from {}", deckLinkName);
            continue;
        }

        cb(deckLinkOutput, std::move(deckLinkName));
    }
}

}  // namespace Vape
