#include "fundamentals/deserializablestore.hpp"

namespace Vape {

DeserializableStore::DeserializableStore()
{
    this->objectRegistered.connect([this](IDeserializable *obj) {
        auto pendingValueIt = this->pendingValues.find(obj->path);
        if (pendingValueIt == this->pendingValues.end()) {
            // No pending value available for an object at this path
            return;
        }

        const PendingValue &pendingValue = pendingValueIt->second;

        // Consume the pending value
        obj->Deserialize(pendingValue.value, pendingValue.userData);
        this->pendingValues.erase(pendingValueIt);
    });
}

DeserializableStore::UpdateReturnValue
DeserializableStore::UpdateValue(const std::string &path, const rapidjson::Value &value,
                                 bool honorRemoteReadOnly, boost::any userData) const
{
    // [CORE UPDATE VALUE FROM BRIDGE 11] Find underlying "Deserializable" object at given path, and modify it
    auto *obj = this->getObject(path);

    if (obj == nullptr) {
        return UpdateReturnValue::ObjectDoesNotExist;
    }

    if (honorRemoteReadOnly && obj->remoteReadOnly) {
        return UpdateReturnValue::RemoteReadOnlyError;
    }

    obj->Deserialize(value, userData);

    return UpdateReturnValue::Success;
}

DeserializableStore::UpdateReturnValue
DeserializableStore::UpdateMemberValue(const std::string &path, const std::string &member,
                                       const rapidjson::Value &value, bool honorRemoteReadOnly,
                                       boost::any userData) const
{
    auto *obj = this->getObject(path);

    if (obj == nullptr) {
        return UpdateReturnValue::ObjectDoesNotExist;
    }

    if (honorRemoteReadOnly && obj->remoteReadOnly) {
        return UpdateReturnValue::RemoteReadOnlyError;
    }

    obj->DeserializeMember(member, value, userData);

    return UpdateReturnValue::Success;
}

}  // namespace Vape
