#include "fundamentals/environment-args.hpp"

#include <boost/algorithm/string.hpp>

#include <cstdlib>
#include <stdexcept>
#include <unordered_set>

namespace Vape::Environment {

namespace {

constexpr auto MAX_PORT_VALUE = 65535;

const std::unordered_set<std::string> TRUTHY_VALUES = {"1", "true"};

}  // namespace

std::string
ParseString(const char *environmentVariable, std::string defaultValue)
{
    const char *stringValue = std::getenv(environmentVariable);
    if (stringValue == nullptr) {
        return defaultValue;
    }

    return stringValue;
}

uint16_t
ParsePort(const char *environmentVariable, const uint16_t &defaultValue)
{
    const char *stringValue = std::getenv(environmentVariable);
    if (stringValue == nullptr) {
        return defaultValue;
    }

    try {
        auto parsedPort = std::stoul(stringValue);
        if (parsedPort > MAX_PORT_VALUE) {
            return defaultValue;
        }

        return static_cast<uint16_t>(parsedPort);
    } catch (const std::invalid_argument &e) {
        return defaultValue;
    } catch (const std::out_of_range &e) {
        return defaultValue;
    }

    return defaultValue;
}

int
ParseInt(const char *environmentVariable, const int &defaultValue)
{
    const char *stringValue = std::getenv(environmentVariable);
    if (stringValue == nullptr) {
        return defaultValue;
    }

    try {
        return std::stoi(stringValue);
    } catch (const std::invalid_argument &e) {
        return defaultValue;
    } catch (const std::out_of_range &e) {
        return defaultValue;
    }

    return defaultValue;
}

bool
ParseBool(const char *environmentVariable, const bool &defaultValue)
{
    const char *stringValue = std::getenv(environmentVariable);
    if (stringValue == nullptr) {
        return defaultValue;
    }

    std::string stdStringValue(stringValue);

    boost::algorithm::to_lower(stdStringValue);

    return TRUTHY_VALUES.find(stdStringValue) != TRUTHY_VALUES.end();
}

}  // namespace Vape::Environment
