#include "fundamentals/gpu.hpp"

#include <GL/glew.h>
#include <d3d11.h>
#include <d3d9.h>

namespace Vape {

bool
GetOpenGLTextureParams(const VapourPixelFormat pixelFormat, OpenGLTextureParams &params)
{
    switch (pixelFormat) {
        case VapourPixelFormat_RGB8:
            params = {GL_RGB8, GL_RGB, GL_UNSIGNED_BYTE, false};
            break;
        case VapourPixelFormat_RGBA8:
            params = {GL_RGBA8, GL_RGBA, GL_UNSIGNED_BYTE, false};
            break;
        case VapourPixelFormat_sRGB8:
            params = {GL_SRGB8, GL_RGB, GL_UNSIGNED_BYTE, true};
            break;
        case VapourPixelFormat_sRGBA8:
            params = {GL_SRGB8_ALPHA8, GL_RGBA, GL_UNSIGNED_BYTE, true};
            break;
        case VapourPixelFormat_sBGR8:
            params = {GL_SRGB8, GL_BGR, GL_UNSIGNED_BYTE, true};
            break;
        case VapourPixelFormat_sBGRA8:
            params = {GL_SRGB8_ALPHA8, GL_BGRA, GL_UNSIGNED_BYTE, true};
            break;
        case VapourPixelFormat_RGBA16:
            params = {GL_RGBA16, GL_RGBA, GL_FLOAT, false};
            break;
        case VapourPixelFormat_UYVY:
        case VapourPixelFormat_I420:
        case VapourPixelFormat_R8:
            params = {GL_R8, GL_RED, GL_UNSIGNED_BYTE, false};
            break;
        case VapourPixelFormat_R16:
            params = {GL_R16, GL_RED, GL_FLOAT, false};
            break;
        default:
            return false;
    }

    return true;
}

bool
GetDX9TextureParams(const VapourPixelFormat pixelFormat, DX9TextureParams &params)
{
    switch (pixelFormat) {
        case VapourPixelFormat_RGBA8:
            params = {D3DFMT_A8B8G8R8};
            break;
        case VapourPixelFormat_RGBA16:
            params = {D3DFMT_A16B16G16R16F};
            break;
        default:
            return false;
    }

    return true;
}

bool
GetDX11TextureParams(const VapourPixelFormat pixelFormat, DX11TextureParams &params)
{
    switch (pixelFormat) {
        case VapourPixelFormat_RGBA8:
            params = {DXGI_FORMAT_R8G8B8A8_UNORM};
            break;
        case VapourPixelFormat_RGBA16:
            params = {DXGI_FORMAT_R16G16B16A16_FLOAT};
            break;
        case VapourPixelFormat_UYVY:
        case VapourPixelFormat_I420:
        case VapourPixelFormat_R8:
            params = {DXGI_FORMAT_R8_UNORM};
            break;
        case VapourPixelFormat_R16:
            params = {DXGI_FORMAT_R16_FLOAT};
            break;
        default:
            return false;
    }

    return true;
}

size_t
SizeFromPixelFormat(Resolution virtualResolution, VapourPixelFormat pixelFormat)
{
    size_t pixels = virtualResolution.x() * virtualResolution.y();

    switch (pixelFormat) {
        case VapourPixelFormat_RGB8:
        case VapourPixelFormat_sBGR8:
        case VapourPixelFormat_sRGB8:
            pixels *= 3;
            break;
        case VapourPixelFormat_RGBA8:
        case VapourPixelFormat_sRGBA8:
        case VapourPixelFormat_sBGRA8:
            pixels *= 4;
            break;
        case VapourPixelFormat_RGBA16:
            pixels *= 8;
            break;
        case VapourPixelFormat_UYVY:
        case VapourPixelFormat_R16:
            pixels *= 2;
            break;
        case VapourPixelFormat_I420:
            pixels = pixels + pixels / 2;
            break;
        case VapourPixelFormat_R8:
            break;
        default:
            assert(false);
    }

    return pixels;
}

int
ChannelCountFromPixelFormat(VapourPixelFormat pixelFormat)
{
    switch (pixelFormat) {
        case VapourPixelFormat_sRGBA8:
        case VapourPixelFormat_sBGRA8:
        case VapourPixelFormat_RGBA16:
            return 4;
        case VapourPixelFormat_RGB8:
        case VapourPixelFormat_sBGR8:
        case VapourPixelFormat_sRGB8:
        case VapourPixelFormat_RGBA8:
            return 3;
        case VapourPixelFormat_R16:
        case VapourPixelFormat_R8:
            return 1;
        case VapourPixelFormat_UYVY:
        case VapourPixelFormat_I420:
        default:
            assert(false);
    }

    return 0;
}

}  // namespace Vape
