#include "fundamentals/iapplication.hpp"

#include "debug/log.hpp"
#include "fundamentals/ipollable.hpp"

#include <cassert>
#include <functional>
#include <thread>

#ifdef _WIN32
#include <Windows.h>
#elif defined(__linux__)
#include <csignal>
#endif

namespace Vape {

namespace {

static std::atomic_bool applicationExiting = false;

#ifdef _WIN32
BOOL
CtrlHandler(DWORD fdwCtrlType)
{
    applicationExiting = true;

    return TRUE;
}
#elif defined(__linux__)

const char *FIRST_EXIT_MESSAGE  = "First exit signal received, attempting clean exit";
const char *SECOND_EXIT_MESSAGE = "Second exit signal received, killing app";

void
CtrlHandler(int signalNumber)
{
    static bool first{true};

    switch (signalNumber) {
        case SIGINT:
        case SIGTERM:
            if (first) {
                Log::RawPrintln(FIRST_EXIT_MESSAGE);
                applicationExiting = true;
                return;
            }

            Log::RawPrintln(SECOND_EXIT_MESSAGE);
            exit(1);
            return;

            break;
    }
}
#endif

}  // namespace

void
IApplication::Start()
{
    this->HandleConsoleExit();

    // This could be an event dispatcher, but we have no need for one at the moment
    while (!applicationExiting) {
        this->Poll();

        std::this_thread::sleep_for(this->eventLoopSleepDuration);
    }
}

void
IApplication::Exit()
{
    applicationExiting = true;
}

void
IApplication::Poll()
{
    for (IPollable &service : this->pollable) {
        service.Poll();
    }
}

/// Public Static
bool
IApplication::IsExiting()
{
    return applicationExiting;
}

void
IApplication::HandleConsoleExit()
{
#ifdef _WIN32
    // Stop CTRL+C and other similar shortcuts from being automatically handled
    DWORD dwMode = 0x0;
    GetConsoleMode(GetStdHandle(STD_INPUT_HANDLE), &dwMode);
    dwMode &= ~ENABLE_PROCESSED_INPUT;
    dwMode &= ~ENABLE_QUICK_EDIT_MODE;
    SetConsoleMode(GetStdHandle(STD_INPUT_HANDLE), dwMode);

    if (!SetConsoleCtrlHandler((PHANDLER_ROUTINE)CtrlHandler, TRUE)) {
        throw std::runtime_error("Unable to set Console Exit handler");
    }
#elif defined(__linux__)
    // Handle CTRL+C sent from
    signal(SIGINT, CtrlHandler);
    signal(SIGTERM, CtrlHandler);
#endif
}

}  // namespace Vape
