#include "fundamentals/rapidjson-helpers.hpp"

#include "debug/log.hpp"

namespace Vape::rj {

void
SetResponseMessageTarget(const rapidjson::Document &message, rapidjson::Document &response)
{
    // If the message has a MessageSource, we copy that into the MessageTarget key
    if (message.HasMember("MessageSource")) {
        const rapidjson::Value &value = message["MessageSource"];
        if (value.IsString()) {
            std::string str = value.GetString();
            Set(response, "MessageTarget", str);
            // {"MessageTarget": ...}
        }
    }
}

rapidjson::Document
CreateResponse(const rapidjson::Document &root)
{
    // Create a rapidjson document with object type
    rapidjson::Document response(rapidjson::kObjectType);
    // {}

    Set(response, "Type", "Publish");
    // {"Type": "Publish"}

    if (root.HasMember("Nonce")) {
        // If the root message has a Nonce, we copy that
        const rapidjson::Value &nonce = root["Nonce"];
        if (nonce.IsString()) {
            std::string str = nonce.GetString();
            Set(response, "Nonce", str);
            // {"Nonce": ...}
        }
    }

    return response;
}

rapidjson::Document
CreateTargetedResponse(const rapidjson::Document &root)
{
    // Create a rapidjson document with object type
    rapidjson::Document response(rapidjson::kObjectType);
    // {}

    Set(response, "Type", "Publish");
    // {"Type": "Publish"}

    // If the root message has a Topic, we copy that and append ":Response"
    if (root.HasMember("Topic")) {
        const rapidjson::Value &value = root["Topic"];
        if (value.IsString()) {
            std::string str = value.GetString();
            Set(response, "Topic", str + ":Response");
            // {"Topic": ...}
        }
    }

    SetResponseMessageTarget(root, response);

    return response;
}

rapidjson::Value &
AddMember(rapidjson::Value &object, const char *key, rapidjson::Value &&value,
          rapidjson::Document::AllocatorType &a)
{
    object.AddMember(rapidjson::Value(key, a).Move(),  // Key
                     value,                            // value
                     a);

    return object[key];
}

bool
Parse(rapidjson::Document &document, const std::string &str)
{
    rapidjson::ParseResult ok = document.Parse(str.c_str());

    return static_cast<bool>(ok);
}

std::string
StringifyPretty(const rapidjson::Value &v)
{
    rapidjson::StringBuffer buffer;
    rapidjson::PrettyWriter<rapidjson::StringBuffer> writer(buffer);
    v.Accept(writer);

    return std::string(buffer.GetString());
}

rapidjson::SchemaDocument
MakeJSONSchema(const char *schemaJSON)
{
    rapidjson::Document sd;
    if (sd.Parse(schemaJSON).HasParseError()) {
        throw std::runtime_error("Unable to parse JSON schema");
    }

    return rapidjson::SchemaDocument(sd);
}

void
PrintSchemaError(const std::string &h, rapidjson::SchemaValidator &validator)
{
    std::string keyword(validator.GetInvalidSchemaKeyword());
    rapidjson::StringBuffer sb;

    if (keyword == "additionalProperties") {
        validator.GetInvalidDocumentPointer().StringifyUriFragment(sb);
        Log::TError(h, "Unwanted key found: {}", sb.GetString());
        return;
    }

    if (keyword == "required") {
        Log::TError(h, "Missing required key");
        return;
    }

    if (keyword == "type") {
        validator.GetInvalidDocumentPointer().StringifyUriFragment(sb);
        Log::TError(h, "Key is wrong type: {}", sb.GetString());
        return;
    }

    validator.GetInvalidSchemaPointer().StringifyUriFragment(sb);
    Log::TError(h, "Error occured in schema: {}", sb.GetString());
    sb.Clear();
    validator.GetInvalidDocumentPointer().StringifyUriFragment(sb);
    Log::TError(h, "Invalid keyword: {}, document: {}", validator.GetInvalidSchemaKeyword(),
                sb.GetString());
}

}  // namespace Vape::rj
