#include "fundamentals/websocket-helpers.hpp"

#include "fundamentals/string-helpers.hpp"

#include <boost/algorithm/string.hpp>
#include <boost/range/combine.hpp>

namespace Vape::ws {

bool
MatchesPath(const std::string &listenerPath, const std::string &messagePath)
{
    using namespace boost;

    // Empty listener path          = Match all message paths
    // Forward slash listener path  = Match all message paths
    if (listenerPath.empty() || listenerPath == "/") {
        return true;
    }

    // Normal paths MUST start with /
    if (!boost::algorithm::starts_with(listenerPath, "/")) {
        return false;
    }

    // Early out truthly for matching paths
    if (listenerPath == messagePath) {
        return true;
    }

    // Any listening path that ends with a forward slash means you're interested in everything under the given path

    auto listenerTokens = Split(listenerPath, "/");
    auto messageTokens  = Split(messagePath, "/");

    if (listenerTokens.size() > messageTokens.size()) {
        // Early out for obvious non-matches, i.e:
        // Listener: /a/b/c
        // Message:  /a/b
        return false;
    }

    bool greedy = boost::algorithm::ends_with(listenerPath, "/");

    if (!greedy) {
        if (listenerTokens.size() < messageTokens.size()) {
            // Early out for obvious non-matches, i.e:
            // Listener: /a/b
            // Message:  /a/b/c
            return false;
        }
    }

    if (listenerTokens.size() == messageTokens.size()) {
        if (greedy) {
            // Early out for obvious non-matches, i.e:
            // Listener: /a/b/
            // Message:  /a/b
            // A greedy listener path matches anything BEYOND its own path
            return false;
        }
    }

    assert(listenerTokens.size() <= messageTokens.size());

    std::vector<std::string>::size_type index              = 0;
    std::vector<std::string>::size_type listenerTokensSize = listenerTokens.size();

    for (auto &&items : boost::combine(listenerTokens, messageTokens)) {
        ++index;

        const auto &listenerToken = items.get<0>();
        const auto &messageToken  = items.get<1>();
        if (listenerToken == "*") {
            continue;
        }

        if (listenerToken != messageToken) {
            return false;
        }

        if (index >= listenerTokensSize) {
            break;
        }
    }

    return true;
}

}  // namespace Vape::ws
