#include <assert.h>
#include <fundamentals/signal-group.hpp>
#include <fundamentals/signal-provider.hpp>
#include <gtest/gtest.h>

using namespace Vape;

class TestSignalProvidingClass
{
public:
    Signal::Provider<int> providerA;
    Signal::Provider<float, int> providerB;

    void
    TriggerA(int a)
    {
        this->providerA.Emit(a);
    }

    void
    TriggerB(float a, int b)
    {
        this->providerB.Emit(a, b);
    }
};

TEST(Signals, DirectListener)
{
    TestSignalProvidingClass p;

    int a          = 0;
    auto listenerA = p.providerA.AddDirectListener([&a](int v) { a += v; });

    EXPECT_EQ(a, 0);

    p.TriggerA(15);

    EXPECT_EQ(a, 15);
}

TEST(Signals, IndirectListener)
{
    TestSignalProvidingClass p;

    Signal::Group groupA;
    Signal::Group groupB;

    int a          = 0;
    int b          = 0;
    auto listenerA = p.providerA.AddIndirectListener(groupA, [&a](int v) { a += v; });
    auto listenerB = p.providerA.AddIndirectListener(groupB, [&b](int v) { b += v; });

    EXPECT_EQ(a, 0);
    EXPECT_EQ(b, 0);

    p.TriggerA(15);

    EXPECT_EQ(a, 0);
    EXPECT_EQ(b, 0);

    groupA.Flush();

    EXPECT_EQ(a, 15);
    EXPECT_EQ(b, 0);

    groupB.Flush();

    EXPECT_EQ(a, 15);
    EXPECT_EQ(b, 15);
}

TEST(Signals, ListenerDestruction)
{
    TestSignalProvidingClass p;

    Signal::Group groupA;

    auto listenerA = p.providerA.AddDirectListener([](int) {});
    auto listenerB = p.providerA.AddIndirectListener(groupA, [](int) {});
    auto listenerC = p.providerA.AddIndirectListener(groupA, [](int) {});

    EXPECT_EQ(p.providerA.ApproxListenerCount(), 3);

    listenerB.reset();

    EXPECT_EQ(p.providerA.ApproxListenerCount(), 3);

    // A dead listener is removed from list after the next signal is emitted
    p.TriggerA(0);

    EXPECT_EQ(p.providerA.ApproxListenerCount(), 2);
}

TEST(Signals, DeadProvider)
{
    std::shared_ptr<Signal::Listener<int>> listenerA;
    Signal::Group groupA;
    int a = 0;

    {
        TestSignalProvidingClass p;

        listenerA = p.providerA.AddIndirectListener(groupA, [&a](int v) { a += v; });

        EXPECT_EQ(a, 0);

        p.TriggerA(1);
    }

    EXPECT_EQ(a, 0);
    groupA.Flush();
    EXPECT_EQ(a, 1);
}

TEST(Signals, DeadListener)
{
    Signal::Group groupA;
    int a = 0;

    {
        TestSignalProvidingClass p;

        auto listenerA = p.providerA.AddIndirectListener(groupA, [&a](int v) { a += v; });

        EXPECT_EQ(a, 0);

        p.TriggerA(1);
    }

    EXPECT_EQ(a, 0);
    groupA.Flush();
    EXPECT_EQ(a, 0);
}
