#include "fundamentals/environment-args.hpp"

#include <gtest/gtest.h>

#include <cstdlib>
#include <unordered_set>

using namespace Vape::Environment;

TEST(EnvironmentArgs, ParseString)
{
    // existant and set
    constexpr const char *en1 = "__TEST_STRING1";
    setenv(en1, "foo", 1);

    // existant and empty
    constexpr const char *en2 = "__TEST_STRING2";
    setenv(en2, "", 1);

    // nonexistant
    constexpr const char *en3 = "__TEST_STRING3";

    // existant and set environment variable should return the set value
    EXPECT_EQ(ParseString(en1, "default"), "foo");

    // existant and empty environment variable should return the empty value
    EXPECT_EQ(ParseString(en2, "default"), "");

    // nonexistant environment variable should return the default value
    EXPECT_EQ(ParseString(en3, "default"), "default");
}

TEST(EnvironmentArgs, ParsePort)
{
    // existant and set
    constexpr const char *en1 = "__TEST_PORT1";
    setenv(en1, "443", 1);

    // existant and empty
    constexpr const char *en2 = "__TEST_PORT2";
    setenv(en2, "", 1);

    // nonexistant
    constexpr const char *en3 = "__TEST_PORT3";

    // existant and out of bounds negative
    constexpr const char *en4 = "__TEST_PORT4";
    setenv(en4, "-50", 1);

    // existant and out of bounds
    constexpr const char *en5 = "__TEST_PORT5";
    setenv(en5, "999999", 1);

    // existant and 0
    constexpr const char *en6 = "__TEST_PORT6";
    setenv(en6, "0", 1);

    // existant and set environment variable should return the set value
    EXPECT_EQ(ParsePort(en1, 1337), 443);

    // existant and empty environment variable should return the default value (as and empty string does not parse into an integer value)
    EXPECT_EQ(ParsePort(en2, 1337), 1337);

    // nonexistant environment variable should return the default value
    EXPECT_EQ(ParsePort(en3, 1337), 1337);

    // existant environment variable but out of bounds (negative) should return the default value
    EXPECT_EQ(ParsePort(en4, 1337), 1337);

    // existant environment variable but out of bounds should return the default value
    EXPECT_EQ(ParsePort(en5, 1337), 1337);

    // existant and set environment variable should return the set value
    EXPECT_EQ(ParsePort(en6, 1337), 0);
}

TEST(EnvironmentArgs, ParseInt)
{
    // existant and set
    constexpr const char *en1 = "__TEST_INT1";
    setenv(en1, "443", 1);

    // existant and empty
    constexpr const char *en2 = "__TEST_INT2";
    setenv(en2, "", 1);

    // nonexistant
    constexpr const char *en3 = "__TEST_INT3";

    // existant and out of bounds negative
    constexpr const char *en4 = "__TEST_INT4";
    setenv(en4, "-9999999999999999999999999999", 1);

    // existant and out of bounds
    constexpr const char *en5 = "__TEST_INT5";
    setenv(en5, "9999999999999999999999999999", 1);

    // existant and 0
    constexpr const char *en6 = "__TEST_INT6";
    setenv(en6, "0", 1);

    // existant and set environment variable should return the set value
    EXPECT_EQ(ParseInt(en1, 1337), 443);

    // existant and empty environment variable should return the default value (as and empty string does not parse into an integer value)
    EXPECT_EQ(ParseInt(en2, 1337), 1337);

    // nonexistant environment variable should return the default value
    EXPECT_EQ(ParseInt(en3, 1337), 1337);

    // existant environment variable but out of bounds (negative) should return the default value
    EXPECT_EQ(ParseInt(en4, 1337), 1337);

    // existant environment variable but out of bounds should return the default value
    EXPECT_EQ(ParseInt(en5, 1337), 1337);

    // existant and set environment variable should return the set value
    EXPECT_EQ(ParseInt(en6, 1337), 0);
}

TEST(EnvironmentArgs, ParseBool)
{
    std::unordered_set<const char *> truthyValues{
        "1",                                     //
        "true", "TRUE", "tRuE", "True", "truE",  //
    };

    // existant and set
    constexpr const char *en1 = "__TEST_BOOL1";
    setenv(en1, "1", 1);

    // existant and empty
    constexpr const char *en2 = "__TEST_BOOL2";
    setenv(en2, "", 1);

    // nonexistant
    constexpr const char *en3 = "__TEST_BOOL3";

    // existant and non-truthy-value
    constexpr const char *en4 = "__TEST_BOOL4";
    setenv(en4, "test", 1);

    // existant and non-truthy-value
    constexpr const char *en5 = "__TEST_BOOL5";
    setenv(en5, "123", 1);

    // existant and 0
    constexpr const char *en6 = "__TEST_BOOL6";
    setenv(en6, "0", 1);

    constexpr const char *end = "__TEST_BOOL_DYNAMIC";

    // existant and set environment variable should return the set value
    EXPECT_EQ(ParseBool(en1, true), true);
    EXPECT_EQ(ParseBool(en1, false), true);

    // existant and empty environment variable should return the default value (as a non-truthy value will parse as false)
    EXPECT_EQ(ParseBool(en2, true), false);
    EXPECT_EQ(ParseBool(en2, false), false);

    // nonexistant environment variable should return the default value
    EXPECT_EQ(ParseBool(en3, true), true);
    EXPECT_EQ(ParseBool(en3, false), false);

    // existant environment variable but non-truthy value should return false
    EXPECT_EQ(ParseBool(en4, true), false);
    EXPECT_EQ(ParseBool(en4, false), false);

    // existant environment variable but non-truthy value should return false
    EXPECT_EQ(ParseBool(en5, true), false);
    EXPECT_EQ(ParseBool(en5, false), false);

    // existant environment variable but non-truthy value should return false
    EXPECT_EQ(ParseBool(en6, true), false);
    EXPECT_EQ(ParseBool(en6, false), false);

    for (const auto &truthyValue : truthyValues) {
        setenv(end, truthyValue, 1);
        EXPECT_EQ(ParseBool(end, false), true)
            << "Truthy value '" << truthyValue << "' must parse as true";
    }
}
