#include "fundamentals/websocket-helpers.hpp"

#include <gtest/gtest.h>

using Vape::ws::MatchesPath;

TEST(WSPathMatching, MessagePathEndingWithSlash)
{
    EXPECT_TRUE(MatchesPath("/a/b/c", "/a/b/c/"));
    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/c/d"));
    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/d/ee/f/g/h/i/"));
    EXPECT_TRUE(MatchesPath("/a/b/d/ee/f/g/h/i/", "/a/b/d/ee/f/g/h/i/"));
    EXPECT_TRUE(MatchesPath("/a/b/d/ee/f/g/h/i", "/a/b/d/ee/f/g/h/i/"));
}

TEST(WSPathMatching, Greedy)
{
    EXPECT_TRUE(MatchesPath("/", "/a"));
    EXPECT_FALSE(MatchesPath("/a/", "/a"));
    EXPECT_TRUE(MatchesPath("/a/", "/a/b"));
    EXPECT_FALSE(MatchesPath("/a/", "/b/b"));
    EXPECT_TRUE(MatchesPath("/a/b/c/", "/a/b/c/d"));
    EXPECT_TRUE(MatchesPath("/a/b/", "/a/b/c/d"));
    EXPECT_TRUE(MatchesPath("/a/", "/a/b/c/d"));
}

TEST(WSPathMatching, Exact)
{
    EXPECT_TRUE(MatchesPath("/a/b/c", "/a/b/c"));
    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/cd"));

    EXPECT_FALSE(MatchesPath("/a/b", "/a/b/c/d"));
    EXPECT_FALSE(MatchesPath("/a", "/a/b/c/d"));

    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/d"));
    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/d/"));
    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/d/e"));
    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/d/ee"));
    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/d/ee/f"));
    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/d/ee/f/g"));
    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/d/ee/f/g/h/i"));
    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/d/ee/f/g/h/i/*"));
    EXPECT_FALSE(MatchesPath("/a/b/c", "/a/b/d/ee/f/g/h/dufhgudfhg"));
}

TEST(WSPathMatching, ListenerPathMustStartWithSlash)
{
    // Wildcard listener path = Match all message paths
    EXPECT_FALSE(MatchesPath("asd", "asd"));
    EXPECT_FALSE(MatchesPath("asd", "/asd"));
}

TEST(WSPathMatching, PreviouslyAcceptedNowRemovedSpecial)
{
    // Wildcard listener path = Match all message paths
    EXPECT_FALSE(MatchesPath("*", "/a"));
    EXPECT_FALSE(MatchesPath("*", "/a/b/c"));
    EXPECT_FALSE(MatchesPath("*", ""));
}

TEST(WSPathMatching, Special)
{
    // Empty listener path = Match all message paths
    EXPECT_TRUE(MatchesPath("", "/a"));
    EXPECT_TRUE(MatchesPath("", "/a/b/c"));
    EXPECT_TRUE(MatchesPath("", ""));

    // Forward slash listener path = Match all message paths
    EXPECT_TRUE(MatchesPath("/", "/a"));
    EXPECT_TRUE(MatchesPath("/", "/a/b/c"));
    EXPECT_TRUE(MatchesPath("/", ""));

    EXPECT_FALSE(MatchesPath("//", "a"));
}

TEST(WSPathMatching, AdvancedWildcard)
{
    EXPECT_TRUE(MatchesPath("/a/*/c", "/a/b/c"));
    EXPECT_FALSE(MatchesPath("/a/*/c", "/a/b/d"));
    EXPECT_FALSE(MatchesPath("/a/*/c", "/a/b/c/d"));
    EXPECT_TRUE(MatchesPath("/a/*/c", "/a/x/c"));
    EXPECT_FALSE(MatchesPath("/a/*/c", "/a/x/c/d"));
    EXPECT_FALSE(MatchesPath("/a/*/c", "/a/x/d"));
    EXPECT_TRUE(MatchesPath("/a/*/c/", "/a/b/c/d"));
    EXPECT_TRUE(MatchesPath("/a/*/c/", "/a/x/c/d/e"));

    EXPECT_TRUE(MatchesPath("/a/*/*/d/", "/a/b/c/d/e"));
    EXPECT_FALSE(MatchesPath("/a/*/e", "/a/b/c/d/e"));
    EXPECT_TRUE(MatchesPath("/a/*/*/*/e", "/a/b/c/d/e"));
}

// Empty target matching should ONLY evaluate to true if:
// Listener is empty
// Listener is /
TEST(WSPathMatching, EmptyTarget)
{
    EXPECT_FALSE(MatchesPath("/a/*/*/d/", ""));
    EXPECT_FALSE(MatchesPath("/a/*/e", ""));
    EXPECT_FALSE(MatchesPath("/a/*/*/*/e", ""));
    EXPECT_FALSE(MatchesPath("/a/b/c", ""));

    EXPECT_TRUE(MatchesPath("/", ""));
    EXPECT_TRUE(MatchesPath("", ""));
}
