import { Leaderboard, Token } from '../interfaces/interfaces';

import axios, {
  AxiosRequestConfig,
  AxiosResponse,
} from 'axios';
import { ValidateJSON } from './validator';

const VapourGameClientID: string | undefined = process.env.VAPOUR_GAMEAPI_CLIENTID;
const VapourGameClientSecret: string | undefined = process.env.SECRET_VAPOUR_GAMEAPI_CLIENTSECRET;

const b64auth = Buffer.from(`${VapourGameClientID}:${VapourGameClientSecret}`).toString('base64');

const Host = 'https://testapidomain';
const TokenEndpoint = `${Host}/account/api/oauth/token`;
const LeaderboardEndpoint = (i: string, w: string, p: number) =>
  `${Host}/api/v1/leaderboards/Game/${i}/${w}/?page=${p}`;
const TokenExpirationMargin = 15 * 60 * 1000; // 15 minutes

function CheckToken(token: Token) {
  const expires = new Date(token.expires_at).getTime();
  const now = new Date().getTime();

  return expires - TokenExpirationMargin > now;
}

export class GameAPI {
  static IsAuthorizationSet(): boolean {
    return !(!VapourGameClientID || !VapourGameClientSecret);
  }

  private static async GetToken(): Promise<Token> {
    console.log('Get new token');

    const formData = new FormData();
    formData.set('grant_type', 'client_credentials');
    formData.set('token_type', 'eg1');

    const options: AxiosRequestConfig = {
      method: 'post',
      url: TokenEndpoint,
      responseType: 'json',
      headers: {
        'Content-Type': 'multipart/form-data',
        'Authorization': `Basic ${b64auth}`,
      },
      data: formData,
    };

    const tokenResponse: AxiosResponse<Token> = await axios(options);
    ValidateJSON(tokenResponse.data, '/Token');

    return tokenResponse.data;
  }

  private activeToken?: Token;

  async GetLeaderboard(token: Token, eventId: string, eventWindow: string) {
    const options: AxiosRequestConfig = {
      url: LeaderboardEndpoint(eventId, eventWindow, 0),
      responseType: 'json',
      headers: {
        Authorization: `Bearer ${token.access_token}`,
      },
    };

    const leaderboardResponse: AxiosResponse<Leaderboard> = await axios(options);
    ValidateJSON(leaderboardResponse.data, '/Leaderboard');
    return leaderboardResponse.data;
  }

  async GetToken(): Promise<Token> {
    if (!VapourGameClientID || !VapourGameClientSecret) {
      throw new Error('Client ID or secret not set');
    }

    if (this.activeToken && CheckToken(this.activeToken)) {
      // We already have a valid token
      return this.activeToken;
    }

    this.activeToken = undefined;
    const newToken = await GameAPI.GetToken();

    if (!CheckToken(newToken)) {
      throw new Error('New token is invalid');
    }

    this.activeToken = newToken;

    return this.activeToken;
  }
}
