import { TypeDesc } from 'vapour-connect/dist';
import { PublishData } from 'vapour-connect/dist/interfaces';
import { TypeInstanceWS } from 'vapour-connect/dist/type';
import { VapourTopicListener } from 'vapour-service-base/dist/topiclistener';
import { VapourTopicProvider } from 'vapour-service-base/dist/topicprovider';

import { Streamer } from './streamer';
import { SyncList } from './synclist';
import { GameApiService } from '../index';
import { Leaderboard, Token, UserChannel, UserChannelData } from '../interfaces/interfaces';
import { SyncListFactory } from '../factory/synclist';
import { ValidateJSON } from './validator';

export class GameLeaderboard {
  allProvider: VapourTopicProvider;
  focusProvider: VapourTopicProvider;

  selectedStreamerSetting: TypeInstanceWS;

  streamers: SyncList<UserChannel, Streamer>;

  constructor(public service: GameApiService, private basePath: string) {
    this.selectedStreamerSetting = service.typeManager.GetConnectedTypeInstance(
      TypeDesc.String,
      basePath + '/selectedStreamer'
    );

    this.allProvider = service.AddProvider('GameTitleAllLeaderboards', () =>
      this.GetAllLeaderboards()
    );
    this.focusProvider = service.AddProvider('GameTitleStreamerFocus', () =>
      this.GetStreamerLeaderboard()
    );

    service.vapour.ws.subscribe('UserChannelData', data =>
      this.UpdateUserList(data?.Data as UserChannelData)
    );

    const cbTriggerUpdate = {
      ValueUpdated: () => {
        this.focusProvider.TriggerUpdate();
      },
    };

    this.selectedStreamerSetting.Subscribe(cbTriggerUpdate);

    const addStreamer = (data: UserChannel) => {
      return new Streamer(this.service.api, data);
    };

    const removeStreamer = (streamer: Streamer) => {
      streamer.Destroy();
    };

    this.streamers = SyncListFactory.Create<UserChannel, Streamer>(addStreamer, removeStreamer);
  }

  UpdateUserList(userData: UserChannelData) {
    ValidateJSON(userData, '/UserChannelData');

    // Trim data to make sure we only use the relevant fields to update the list
    const trimmedData: UserChannel[] = userData.Channels.map(
      (u): UserChannel => {
        return {
          'Channel Name': u['Channel Name'],
          'In Game Display Name': u['In Game Display Name'],
        };
      }
    );

    if (this.streamers.Sync(trimmedData)) {
      this.focusProvider.TriggerUpdate();
      this.allProvider.TriggerUpdate();
    }

    return undefined;
  }

  async Update() {
    const token: Token = await this.service.api.GetToken();

    const updates: Array<Promise<boolean>> = this.streamers.GetElements().map(s => s.Update(token));
    const anyUpdate: boolean[] = await Promise.all(updates);

    if (anyUpdate.indexOf(true) !== -1) {
      this.allProvider.TriggerUpdate();
      this.focusProvider.TriggerUpdate();
    }
  }

  GetAllLeaderboards(): PublishData {
    return {
      Leaderboard: this.streamers.GetElements().map(s => s.GetLeaderboardData()),
    };
  }

  GetStreamerLeaderboard(): PublishData {
    let find: Streamer | undefined;
    const list = this.streamers.GetElements();

    const selectedStreamer: string = this.selectedStreamerSetting.Get();
    if (selectedStreamer) {
      find = list.find(s => s.channelName === selectedStreamer);
    } else {
      // Return first list item if no match is found
      find = list[0];
    }

    if (find) {
      return {
        Leaderboard: [find.GetFocusData()],
      };
    }

    return {
      Leaderboard: [],
    };
  }
}
