import { expect } from 'chai';
import { describe, it } from 'mocha';
import { SleeperFactory } from '../factory/sleeper';
import { RequestLoop, RequestLoopResult } from './request-loop';

describe('RequestLoop', async () => {
  it('Should expect RequestLoop to run with expected intervals', async () => {
    let counter: number = 0;
    let causeError = false;
    let lastErrorState = false;

    const reportFn = (res: RequestLoopResult) => {
      lastErrorState = res.error;
    };

    const fn = async () => {
      counter++;

      if (causeError) {
        throw new Error('Dummy error');
      }
    };

    const requestLoop: RequestLoop = new RequestLoop(fn, reportFn, 50);
    const run = requestLoop.Run();

    // Timer inactive and should not run function
    await SleeperFactory.Create(100).Await();
    expect(counter).to.equal(0);
    expect(lastErrorState).to.equal(false);

    // Manually trigger function
    requestLoop.TriggerOnce();
    await SleeperFactory.Create(100).Await();
    expect(counter).to.equal(1);
    expect(lastErrorState).to.equal(false);

    // Activate timer
    requestLoop.SetActive(true);
    await SleeperFactory.Create(500).Await();
    expect(lastErrorState).to.equal(false);

    // Stop timer
    requestLoop.SetActive(false);
    expect(counter).to.greaterThan(7);
    expect(lastErrorState).to.equal(false);

    // Reset counter value and test that manual trigger calls the function once
    counter = 0;
    causeError = true;
    requestLoop.TriggerOnce();
    await SleeperFactory.Create(200).Await();
    expect(counter).to.equal(1);
    expect(lastErrorState).to.equal(true);

    // Stop loop and wait for it to finish
    requestLoop.Stop();
    await run;
  });
});
