import { isEqual } from 'lodash';

export type CreateListElement<DataType, ElemType> = (data: DataType) => ElemType;
export type RemoveListElement<ElemType> = (elem: ElemType) => void;

interface ListElem<DataType, ElemType> {
  data: DataType;
  elem: ElemType;
}

export class SyncList<DataType, ElemType> {
  private list: Array<ListElem<DataType, ElemType>> = [];

  constructor(
    private addCb: CreateListElement<DataType, ElemType>,
    private removeCB: RemoveListElement<ElemType>
  ) {}

  Sync(newList: DataType[]): boolean {
    const len: number = Math.max(this.list.length, newList.length);
    let wasUpdated: boolean = false;

    for (let i = 0; i < len; ++i) {
      const a: DataType | undefined = newList[i];
      const b: ListElem<DataType, ElemType> | undefined = this.list[i];

      if (a !== undefined && b === undefined) {
        // Check if slot was empty but now should be filled
        this.list.push({ data: a, elem: this.addCb(a) });
        wasUpdated = true;
      } else if (a === undefined && b !== undefined) {
        // Check if slot was filled but now should be empty
        this.removeCB(b.elem);
        wasUpdated = true;
      } else if (a !== undefined && b !== undefined) {
        // Check if data matches for existing element in slot
        if (!isEqual(a, b.data)) {
          this.removeCB(b.elem);
          const newElem: ElemType = this.addCb(a);
          this.list[i] = { data: a, elem: newElem };
          wasUpdated = true;
        }
      }
    }

    // Adjust length of list to match data
    if (this.list.length !== newList.length) {
      this.list.length = newList.length;
      wasUpdated = true;
    }

    return wasUpdated;
  }

  GetElements(): ElemType[] {
    return this.list.map(l => l.elem);
  }

  Clear(): boolean {
    return this.Sync([]);
  }
}
