package permissions

// Effect is the valid permission effect constants
type Effect string

const (
	// EffectAllow is the "Allow" effect on permissions.
	EffectAllow Effect = "Allow"
)

// Valid checks whether the specified effect type is a valid constant
func (e Effect) Valid() bool {
	return e == EffectAllow
}

// String is a stringer
func (e Effect) String() string {
	return string(e)
}

// Scope is the different types of scopes each permission can have
type Scope string

const (
	// ScopeSubscribe dictates that they can subscribe to this path
	ScopeSubscribe Scope = "Subscribe"
	// ScopePublish dictates that they can publish messages to this path
	ScopePublish Scope = "Publish"
	// ScopeAdmin isn't something available over core pubsub, this is a special scope that Authenticator endpoints use.
	ScopeAdmin Scope = "Admin"
)

// Valid checks whether the value of the Scope type is valid
func (s Scope) Valid() bool {
	switch s {
	case ScopeSubscribe, ScopePublish, ScopeAdmin:
		return true
	}
	return false
}

// String is a stringer
func (s Scope) String() string {
	return string(s)
}

// PermissionResult is the constant that dictates the result of a permission query
type PermissionResult int

const (
	// PermissionResultNone means no permissions were found
	PermissionResultNone PermissionResult = 0
	// PermissionResultPartial means partial were found
	PermissionResultPartial PermissionResult = 1
	// PermissionResultFull means full were found
	PermissionResultFull PermissionResult = 2
)
