package permissions

import (
	"encoding/json"
	"errors"
	"math"
	"strings"
)

// Path is a parsed path
type Path struct {
	tokenCount  int
	tokens      []string
	greedy      bool
	hasWildcard bool
	// useful to have, when we panic in a goroutine we can log the raw path passed.
	raw string
}

var (
	// ErrPathInvalid is returned when you try to construct a path which has an invalid raw path
	ErrPathInvalid = errors.New("invalid path")
	// ErrPathEmptyToken is returned when you try to construct a path which contains an empty token (ie. "//")
	ErrPathEmptyToken = errors.New("path contains empty token")
	// ErrPathPrefixEmpty is returned when a path is not empty or does not start with /
	ErrPathPrefixEmpty = errors.New("path must start with / or be empty")
)

// NewPath constructs a new path struct
func NewPath(path string) (*Path, error) {
	pathSpl := strings.Split(path, "/")
	if pathSpl[0] != "" {
		return nil, ErrPathPrefixEmpty
	}

	if len(pathSpl) == 1 {
		pathSpl = append(pathSpl, "")
	}

	p := &Path{
		raw: path,
	}

	p.tokenCount = len(pathSpl) - 1

	if pathSpl[p.tokenCount] == "" {
		p.greedy = true
		p.tokenCount--
	}

	p.tokens = pathSpl[1 : p.tokenCount+1]

	for _, token := range p.tokens {
		switch token {
		case "":
			return nil, ErrPathEmptyToken
		case "*":
			p.hasWildcard = true
		}
	}

	return p, nil
}

// Evaluate allows you to evaluate another path against the current path instance.
func (p *Path) Evaluate(req *Path) PermissionResult {
	state := PermissionResultFull
	searchLen := int(math.Min(float64(p.tokenCount), float64(req.tokenCount))) + 1

	for i := 0; i < searchLen; i++ {
		primaryToken := optSliceVal(p.tokens, i)
		reqToken := optSliceVal(req.tokens, i)

		primaryDepl := primaryToken == ""
		reqDepl := reqToken == ""

		if primaryDepl {
			if p.greedy && reqToken != "" {
				return state
			}

			if reqDepl {
				/**
				* If both paths has the same greedy value, current state is valid
				*  t: '/a'   o: '/a'   = Full
				*  t: '/a/'  o: '/a/'  = Full
				*  t: '/a'   o: '/a/'  = None
				*  t: '/a/'  o: '/a'   = None
				 */
				if p.greedy == req.greedy {
					return state
				}
				return PermissionResultNone
			}
		} else if reqDepl {
			// if the requested token is greedy but the primary isn't
			if req.greedy {
				// requested is greedy but the primary contains more specificity, therefore partial match
				return PermissionResultPartial
			}
		} else {
			// Neither depleted.

			if primaryToken == reqToken {
				// exact match
				continue
			}

			if primaryToken == "*" {
				// primary is a wildcard
				continue
			}

			if reqToken == "*" {
				// requested token is matched by a wildcard, therefore a partial match.
				state = PermissionResultPartial
				continue
			}
		}

		return PermissionResultNone
	}

	go panic("failed to reach evaluation for path when evaluating: " + p.raw + "requesting path: " + req.raw)
	return PermissionResultNone
}

// TokenCount returns the number of tokens in the path
func (p *Path) TokenCount() int {
	return p.tokenCount
}

// Tokens returns the string slice of tokens
func (p *Path) Tokens() []string {
	return p.tokens
}

// Greedy returns whether the path is "greedy" (ends with /)
func (p *Path) Greedy() bool {
	return p.greedy
}

// HasWildcard returns whether the path has a wildcard anywhere.
func (p *Path) HasWildcard() bool {
	return p.hasWildcard
}

func (p Path) MarshalJSON() ([]byte, error) {
	return json.Marshal(p.raw)
}

func (p *Path) UnmarshalJSON(data []byte) error {
	var rawString string
	if err := json.Unmarshal(data, &rawString); err != nil {
		return err
	}

	if path, err := NewPath(rawString); err != nil {
		return err
	} else {
		*p = *path
	}

	return nil
}
