package permissions

import (
	"encoding/json"
	"testing"

	"github.com/stretchr/testify/suite"
)

type pathSuite struct {
	suite.Suite
}

func (t *pathSuite) Test_MessagePathEndingWithSlash() {
	t.False(t.testPath("/a/b/c", "/a/b/c/"))
	t.False(t.testPath("/a/b/c", "/a/b/c/d"))
	t.False(t.testPath("/a/b/c", "/a/b/d/ee/f/g/h/i/"))
	t.True(t.testPath("/a/b/d/ee/f/g/h/i/", "/a/b/d/ee/f/g/h/i/"))
	t.False(t.testPath("/a/b/d/ee/f/g/h/i", "/a/b/d/ee/f/g/h/i/"))
}

func (t *pathSuite) Test_Greedy() {
	t.True(t.testPath("/", "/a"))
	t.False(t.testPath("/a/", "/a"))
	t.True(t.testPath("/a/", "/a/b"))
	t.False(t.testPath("/a/", "/b/b"))
	t.True(t.testPath("/a/b/c/", "/a/b/c/d"))
	t.True(t.testPath("/a/b/", "/a/b/c/d"))
	t.True(t.testPath("/a/", "/a/b/c/d"))
}

func (t *pathSuite) Test_Exact() {
	t.True(t.testPath("/a/b/c", "/a/b/c"))
	t.False(t.testPath("/a/b/c", "/a/b/cd"))

	t.False(t.testPath("/a/b", "/a/b/c/d"))
	t.False(t.testPath("/a", "/a/b/c/d"))

	t.False(t.testPath("/a/b/c", "/a/b/d"))
	t.False(t.testPath("/a/b/c", "/a/b/d/"))
	t.False(t.testPath("/a/b/c", "/a/b/d/e"))
	t.False(t.testPath("/a/b/c", "/a/b/d/ee"))
	t.False(t.testPath("/a/b/c", "/a/b/d/ee/f"))
	t.False(t.testPath("/a/b/c", "/a/b/d/ee/f/g"))
	t.False(t.testPath("/a/b/c", "/a/b/d/ee/f/g/h/i"))
	t.False(t.testPath("/a/b/c", "/a/b/d/ee/f/g/h/i/*'"))
	t.False(t.testPath("/a/b/c", "/a/b/d/ee/f/g/h/dufhgudfhg"))
}

func (t *pathSuite) Test_ListenerPathMustStartWithSlash() {
	_, err := NewPath("asd")
	t.Equal(ErrPathPrefixEmpty, err)

	_, err = NewPath("*")
	t.Equal(ErrPathPrefixEmpty, err)

	_, err = NewPath("//")
	t.Equal(ErrPathEmptyToken, err)
}

func (t *pathSuite) Test_Special() {
	t.True(t.testPath("", "/a"))
	t.True(t.testPath("", "/a/b/c"))
	t.True(t.testPath("", ""))

	t.True(t.testPath("/", "/a"))
	t.True(t.testPath("/", "/a/b/c"))
	t.True(t.testPath("/", ""))
}

func (t *pathSuite) Test_AdvancedWildcard() {
	t.True(t.testPath("/a/*/c", "/a/b/c"))
	t.False(t.testPath("/a/*/c", "/a/b/d"))
	t.False(t.testPath("/a/*/c", "/a/b/c/d"))
	t.True(t.testPath("/a/*/c", "/a/x/c"))
	t.False(t.testPath("/a/*/c", "/a/x/c/d"))
	t.False(t.testPath("/a/*/c", "/a/x/d"))
	t.True(t.testPath("/a/*/c/", "/a/b/c/d"))
	t.True(t.testPath("/a/*/c/", "/a/x/c/d/e"))

	t.True(t.testPath("/a/*/*/d/", "/a/b/c/d/e"))
	t.False(t.testPath("/a/*/e", "/a/b/c/d/e"))
	t.True(t.testPath("/a/*/*/*/e", "/a/b/c/d/e"))
}

func (t *pathSuite) Test_EmptyTarget() {
	t.False(t.testPath("/a/*/*/d/", ""))
	t.False(t.testPath("/a/*/e", ""))
	t.False(t.testPath("/a/*/*/*/e", ""))
	t.False(t.testPath("/a/b/c", ""))

	t.True(t.testPath("/", ""))
	t.True(t.testPath("", ""))
}

func (t *pathSuite) Test_Overlap() {
	t.False(t.testPathOverlap("/a/b/*/", "/a/b/q"))
	t.Equal(PermissionResultPartial, t.evaluatePaths("/*/*", "/a/"))
	t.Equal(PermissionResultPartial, t.evaluatePaths("/a/*", "/a/"))
	t.False(t.testPathOverlap("/a/*", "/a/a/"))
	t.Equal(PermissionResultPartial, t.evaluatePaths("/*/a", "/*/*"))
	t.False(t.testPath("/*/a", "/"))
	t.Equal(PermissionResultPartial, t.evaluatePaths("/a/b/c", "/a/*/c"))
	t.Equal(PermissionResultPartial, t.evaluatePaths("/a/b/c/", "/a/*/c/"))
	t.Equal(PermissionResultPartial, t.evaluatePaths("/a/b/", "/a/*/c/"))
	t.True(t.testPath("/a/*/", "/a/*/c"))
}

func (t *pathSuite) Test_CreateGreedy() {
	nonGreedy, err := NewPath("/a/b/c")
	t.NoError(err, "failed to init nonGreedy path")
	t.False(nonGreedy.Greedy())

	greedy, err := NewPath("/a/b/")
	t.NoError(err, "failed to init greedy path")
	t.True(greedy.Greedy())
}

func (t *pathSuite) Test_CreateHasWildcard() {
	noWildcard, err := NewPath("/a/b/c")
	t.NoError(err, "failed to init noWildcard path")
	t.False(noWildcard.HasWildcard())

	wildcard, err := NewPath("/a/b/*/xd")
	t.NoError(err, "failed to init wildcard path")
	t.True(wildcard.HasWildcard())
}

func (t *pathSuite) Test_TokenCount() {
	tokenCount, err := NewPath("/a/b/c")
	t.NoError(err, "failed to init path1")
	t.Equal(3, tokenCount.TokenCount())

	tokenCount, err = NewPath("/a/b/*/d/c")
	t.NoError(err, "failed to init path2")
	t.Equal(5, tokenCount.TokenCount())

	tokenCount, err = NewPath("/a/b")
	t.NoError(err, "failed to init path3")
	t.Equal(2, tokenCount.TokenCount())
}

func (t *pathSuite) Test_PathTokens() {
	path, err := NewPath("/a/b/c")
	t.NoError(err, "failed to init path1")
	tokens := path.Tokens()
	t.Equal("a", tokens[0])
	t.Equal("b", tokens[1])
	t.Equal("c", tokens[2])
	t.Equal(3, len(tokens))

	path, err = NewPath("/b/c/*/e/f")
	t.NoError(err, "failed to init path2")
	tokens = path.Tokens()
	t.Equal("b", tokens[0])
	t.Equal("c", tokens[1])
	t.Equal("*", tokens[2])
	t.Equal("e", tokens[3])
	t.Equal("f", tokens[4])
	t.Equal(5, len(tokens))
}

func (t *pathSuite) Test_PathMarshalsIntoAString() {
	const pathString = "/a/b/c"
	path, err := NewPath(pathString)
	t.NoError(err, "failed to init path")

	marshalledPath, err := json.Marshal(path)
	t.NoError(err, "failed to marshal path")

	marshalledString, err := json.Marshal(pathString)
	t.NoError(err, "failed to marshal string")

	t.Equal(marshalledString, marshalledPath)
}

func (t *pathSuite) Test_PathStringUnmarshalsIntoAPath() {
	const pathString = "/a/b/c"
	marshalledString, err := json.Marshal(pathString)
	t.NoError(err, "failed to marshal string")

	path := &Path{}
	err = json.Unmarshal(marshalledString, path)
	t.NoError(err, "failed to unmarshal path")

	t.Equal(path.raw, pathString)
}

func (t *pathSuite) Test_UnmarshalPathPathMustStartWithASlash() {
	const pathString = "asd"
	marshalledString, err := json.Marshal(pathString)
	t.NoError(err, "failed to marshal string")

	path := &Path{}
	err = json.Unmarshal(marshalledString, path)
	t.Equal(ErrPathPrefixEmpty, err, "failed to unmarshal path")
}

func (t *pathSuite) Test_UnmarshalPathPathMustBeAString() {
	bytes := []byte(`[]`)

	path := &Path{}
	err := json.Unmarshal(bytes, path)
	t.EqualError(err, "json: cannot unmarshal array into Go value of type string", "failed to unmarshal path")
}

func (t *pathSuite) testPath(a, b string) bool {
	return t.evaluatePaths(a, b) == PermissionResultFull
}

func (t *pathSuite) testPathOverlap(a, b string) bool {
	return t.evaluatePaths(a, b) >= PermissionResultPartial
}

func (t *pathSuite) evaluatePaths(a, b string) PermissionResult {
	pathA, err := NewPath(a)
	t.NoError(err, "failed to construct 'a' path: "+a)
	pathB, err := NewPath(b)
	t.NoError(err, "failed to construct 'b' path: "+b)
	return pathA.Evaluate(pathB)
}

func TestPath(t *testing.T) {
	suite.Run(t, &pathSuite{})
}
