package permissions

// Rule is an individual permission rule
type Rule struct {
	Effect Effect
	Topic  string
	Path   *Path
	Scope  Scope
}

// NewRule constructs a new rule from the effect/topic/path/scope
func NewRule(effect Effect, topic, path string, scope Scope) (*Rule, error) {
	parsedPath, err := NewPath(path)
	if err != nil {
		return nil, err
	}

	return &Rule{
		Effect: effect,
		Topic:  topic,
		Path:   parsedPath,
		Scope:  scope,
	}, nil
}

// Evaluate checks if a EvalItem is covered by the rule. Rule is tested for scope, then topic, then path, lowest result is returned.
func (r *Rule) Evaluate(item *EvalItem) PermissionResult {
	if r.Scope != item.Scope {
		return PermissionResultNone
	}

	var topicTest PermissionResult
	switch {
	case r.Topic != "*" && item.Topic == "*":
		topicTest = PermissionResultPartial

	case r.Topic == "*":
		topicTest = PermissionResultFull

	case r.Topic == item.Topic:
		topicTest = PermissionResultFull

	default:
		return PermissionResultNone
	}

	pathTest := r.Path.Evaluate(item.Path)

	if topicTest < pathTest {
		return topicTest
	}
	return pathTest
}
