#include "app.hpp"

#include "./config.hpp"
#include "./status/multistream-audiolevels.hpp"
#include "./status/multistream-decklink.hpp"
#include "./status/multistream-playbackstatus.hpp"
#include "./status/multistream-systemstatus.hpp"
#include "./version.hpp"
#include "api/request.hpp"
#include "debug/log.hpp"
#include "fundamentals/bind.hpp"
#include "fundamentals/helpers.hpp"

#include <chrono>

using namespace std::chrono_literals;

namespace Vape {

Application::Application()
    : PubSub::Service::Base()
    , settingsManager(this->pubsubClient)
    , statusManager(this->pubsubClient)
    , credentialsManager(this->pubsubClient, this->apiClient)
    , player(this->settingsManager, this->signalGroup, this->signalListeners, this->apiClient)
{
    auto subscribe = [this](auto &signal, auto l) {
        this->signalListeners.push_back(signal.AddIndirectListener(this->signalGroup, l));
    };

    this->pubsubClient.SetClientType("multistream");
    this->pubsubClient.SetVersion(Multistream::VERSION);
    this->pubsubClient.SetPathPrefix("/clients/multistream");

    subscribe(this->player.consumerManager.displayModeNamesSignal, [&](const auto &modes) {
        static DisplayModeNames lastModes;
        if (modes != lastModes) {
            this->settingsManager.SetDisplayModes(modes);
            lastModes = modes;
        }
    });

    subscribe(this->player.stopSignal,
              [&](StopReason) { this->statusManager.Reset(this->settingsManager.IsEnabled()); });

    this->throttler.SetDuraionSteps({100ms, 200ms, 200ms, 500ms});
    this->throttler.SetJitterBounds(0, 50);
}

Application::~Application()
{
    this->QuitPlayer();
}

void
Application::StartPlayer()
{
    this->QuitPlayer();
    this->player.quitter.Reset();

    {
        std::unique_lock lock(this->playerStateMutex);
        this->playerState = StopReason::Running;
    }

    {
        // Reset start synchronization variables
        std::unique_lock lock(this->startLock.mutex);
        this->startLock.initialized = false;
    }

    this->playerThread = std::make_unique<std::thread>([this]() {
        StopReason state = this->player.Run(this->settingsManager, this->statusManager,
                                            this->credentialsManager, this->startLock);

        {
            // Set the return value to the varibale that stores the latest player state
            std::unique_lock lock(this->playerStateMutex);
            this->playerState = state;
        }
    });

    {
        // Wait for the player thread to finish initiation before returning
        std::unique_lock<std::mutex> lock(this->startLock.mutex);
        this->startLock.cv.wait(lock, [this] { return this->startLock.initialized; });
    }
}

void
Application::QuitPlayer()
{
    player.quitter.Quit();
    if (this->playerThread) {
        this->playerThread->join();
    }

    this->playerThread.reset();
}

void
Application::Poll()
{
    this->PubSub::Service::Base::Poll();

    if (this->settingsManager.Poll()) {
        this->QuitPlayer();
    }

    this->statusManager.Poll();

    /**
     * Destructive read of the player state
     */

    StopReason state;

    {
        std::unique_lock lock(this->playerStateMutex);
        state = this->playerState;

        if (state == StopReason::Running) {
            // Player is running, no further check needed
            return;
        }

        if (state != StopReason::WaitingForStart) {
            // There is a new exit state that should only be read once,
            // so the value is replaced with the WaitingForStart state
            this->playerState = StopReason::WaitingForStart;
        }
    }

    // There is a new exit state, register it accordingly
    if (state != StopReason::WaitingForStart) {
        if (state == StopReason::None) {
            this->throttler.RegisterSuccess();
        } else {
            this->throttler.RegisterFail();
        }
    }

    // If throttler wants a retry, we retry
    if (this->throttler.ShouldTry()) {
        this->statusManager.Reset(this->settingsManager.IsEnabled());
        this->StartPlayer();
    }
}

Ruleset
Application::CreateRequiredRuleset()
{
    auto prefix = this->pubsubClient.GetResolvedPathPrefix();

    return Ruleset({
        {
            EffectType::Allow,
            "ValueUpdated",
            fS("{}/", prefix),
            ScopeType::Subscribe,
        },
        {
            EffectType::Allow,
            "SetValue",
            fS("/_parameters{}/", prefix),
            ScopeType::Publish,
        },
        {
            EffectType::Allow,
            Topic::MultistreamAudioLevels::topicString,
            prefix,
            ScopeType::Publish,
        },
        {
            EffectType::Allow,
            Topic::MultistreamSystemStatus::topicString,
            prefix,
            ScopeType::Publish,
        },
        {
            EffectType::Allow,
            Topic::MultistreamPlaybackStatus::audioTopicString,
            prefix,
            ScopeType::Publish,
        },
        {
            EffectType::Allow,
            Topic::MultistreamPlaybackStatus::videoTopicString,
            prefix,
            ScopeType::Publish,
        },
        {
            EffectType::Allow,
            Topic::MultistreamDeckLink::topicString,
            prefix,
            ScopeType::Publish,
        },
    });
}

}  // namespace Vape
