#include "./consumer-factory.hpp"

#include "../settings/player.hpp"
#include "./decklink-output.hpp"
#include "./ndi-output.hpp"

namespace Vape {

namespace {

struct IConsumerFactory {
    virtual std::unique_ptr<IVideoOutputConsumer> Create(const ConsumerSettings &s) = 0;

    virtual ~IConsumerFactory() = default;
};

template <typename Consumer>
struct ConsumerFactory : IConsumerFactory {
    std::unique_ptr<IVideoOutputConsumer>
    Create(const ConsumerSettings &s) final
    {
        return std::make_unique<Consumer>(s);
    }
};

std::unique_ptr<IConsumerFactory>
MakeFactory(OutputType outputType)
{
    switch (outputType) {
        case OutputType::NDI: {
            return std::make_unique<ConsumerFactory<NDIOutput>>();
        } break;

        case OutputType::DeckLink: {
            return std::make_unique<ConsumerFactory<DeckLinkOutput>>();
        } break;

        case OutputType::Invalid: {
            return nullptr;
        } break;
    }

    return nullptr;
}

}  // namespace

std::pair<std::unique_ptr<IVideoOutputConsumer>, StopReason>
CreateConsumer(OutputType outputType, const ConsumerSettings &s)
{
    auto factory = MakeFactory(outputType);
    if (!factory) {
        return {nullptr, StopReason::InvalidOutputType};
    }

    return {factory->Create(s), StopReason::None};
}

}  // namespace Vape
