#include "./decklink-allocator.hpp"

#include "debug/log.hpp"
#include "fundamentals/helpers.hpp"

#include <cassert>

namespace Vape {
static int instanceCount = 0;

AVFrameMemoryAllocator::AVFrameMemoryAllocator()
    : h(fS("AVFrameMemoryAllocator[{}]", instanceCount++))
{
    Log::TTrace(h, "Constructor");
}

AVFrameMemoryAllocator::~AVFrameMemoryAllocator()
{
    Log::TTrace(h, "Destructor called with {} allocated frames", this->allocationCount);
    if (this->nextFrame) {
        av_frame_free(&this->nextFrame);
    }
}

// IDeckLinkMemoryAllocator methods
HRESULT STDMETHODCALLTYPE
AVFrameMemoryAllocator::AllocateBuffer(unsigned int bufferSize, void **allocatedBuffer)
{
    std::unique_lock<std::mutex> lock(this->mutex);

    if (this->nextFrame) {
        *allocatedBuffer = this->nextFrame->data[0];

        auto it = this->frames.find(*allocatedBuffer);
        if (it != this->frames.end()) {
            // The underlying buffer is already allocated once (most likely previous frame)
            // Add to frame reference counting and free the new AVFrame
            it->second.count++;
            av_frame_free(&this->nextFrame);
        } else {
            // Store the allocated frame in the map with a starting reference count of 1
            this->frames[*allocatedBuffer] = {this->nextFrame, 1};
        }

        // Pushed AVFrame is used up
        this->nextFrame = nullptr;
    } else {
        // This is a frame that is not allocated via AVFrames and gets allocated normally
        *allocatedBuffer = malloc(bufferSize);
        Log::TTrace(h, "Allocating non AVFrame buffer: {}", (size_t)*allocatedBuffer);

        if (*allocatedBuffer == nullptr) {
            return E_OUTOFMEMORY;
        }
    }

    // For logging
    this->allocationCount++;

    return S_OK;
}

HRESULT STDMETHODCALLTYPE
AVFrameMemoryAllocator::ReleaseBuffer(void *buffer)
{
    std::unique_lock<std::mutex> lock(this->mutex);

    auto it = this->frames.find(buffer);
    if (it != this->frames.end()) {
        auto &record = it->second;

        if (--record.count == 0) {
            AVFrame *toFree = record.frame;
            av_frame_free(&toFree);

            this->frames.erase(it);
        }

    } else {
        Log::TTrace(h, "Releasing non AVFrame buffer: {}", (size_t)buffer);
        free(buffer);
    }

    // For logging
    this->allocationCount--;

    return S_OK;
}

HRESULT STDMETHODCALLTYPE
AVFrameMemoryAllocator::Commit()
{
    Log::TTrace(h, "Commit");
    return S_OK;
}

HRESULT STDMETHODCALLTYPE
AVFrameMemoryAllocator::Decommit()
{
    std::unique_lock<std::mutex> lock(this->mutex);
    Log::TTrace(h, "Decommit called with {} allocated frames", this->allocationCount);
    return S_OK;
}

void
AVFrameMemoryAllocator::PushAVFrame(AVFrame *frame)
{
    std::unique_lock<std::mutex> lock(this->mutex);

    assert(!this->nextFrame);

    this->nextFrame = av_frame_clone(frame);
}
}  // namespace Vape
