#include "./decklink-base.hpp"

#include "debug/log.hpp"
#include "fundamentals/helpers.hpp"

#include <DeckLinkAPIVersion.h>

#include <algorithm>
#include <iostream>
#include <mutex>

using namespace Vape;

std::string
GetDeckLinkName(const ScopedDeckLinkPointer<IDeckLink> &deckLink)
{
    const std::string h("GetDeckLinkName");

    if (!deckLink) {
        return "";
    }

    std::string deckLinkName;
    STRINGOBJ internalName = nullptr;

    // Get model name in BSTR string
    auto result = deckLink->GetDisplayName(&internalName);
    if (FAILED(result)) {
        Log::TError(h, "Failed to get DisplayName");
        return "";
    }

    // Convert internal name string to string
    StringToStdString(internalName, deckLinkName);
    STRINGFREE(internalName);

    auto attributes =
        deckLink.QueryInterface<IDeckLinkProfileAttributes>(IID_IDeckLinkProfileAttributes);

    int64_t topologicalID = 0;

    result = attributes->GetInt(BMDDeckLinkTopologicalID, &topologicalID);
    if (FAILED(result)) {
        Log::TError(h, "Failed to get topological ID of output");
        return "";
    }

    return fS("{} {}", topologicalID, deckLinkName);
}

std::string
GetDecklinkDriverVersion()
{
    ScopedDeckLinkPointer info(CreateDeckLinkAPIInformationInstance());
    if (!info) {
        return "";
    }

    STRINGOBJ dlVersion;

    auto result = info->GetString(BMDDeckLinkAPIVersion, &dlVersion);
    if (FAILED(result)) {
        return "";
    }

    std::string version;
    StringToStdString(dlVersion, version);
    STRINGFREE(dlVersion);

    return version;
}

std::vector<DeckLinkDisplayMode>
GetDisplayModes(const ScopedDeckLinkPointer<IDeckLinkOutput> &deckLinkOutput)
{
    const std::string h("GetDisplayModes");

    std::vector<DeckLinkDisplayMode> list;

    if (!deckLinkOutput) {
        return list;
    }

    IDeckLinkDisplayMode *mode;

    IDeckLinkDisplayModeIterator *iterator = nullptr;
    auto res                               = deckLinkOutput->GetDisplayModeIterator(&iterator);

    if (FAILED(res) || iterator == nullptr) {
        Log::TError(h, "Could not get display mode iterator");
        return list;
    }

    ScopedDeckLinkPointer scopedModeIterator(iterator);

    while (iterator->Next(&mode) == S_OK) {
        BMDFieldDominance field = mode->GetFieldDominance();
        if (field != bmdProgressiveFrame) {
            // Ignore modes that are not progressive
            continue;
        }

        DeckLinkDisplayMode m;

        STRINGOBJ internalName;

        if (FAILED(mode->GetName(&internalName))) {
            Log::TError(h, "Could not get display mode name");
            return list;
        }

        StringToStdString(internalName, m.name);
        STRINGFREE(internalName);

        m.width  = mode->GetWidth();
        m.height = mode->GetHeight();
        m.mode   = mode->GetDisplayMode();
        mode->GetFrameRate(&m.frameDuration, &m.timeScale);

        list.push_back(m);
    }

    return list;
}

bool
CheckDeckLinkAttribute(const ScopedDeckLinkPointer<IDeckLinkProfileAttributes> &attributes,
                       BMDDeckLinkAttributeID attrID)
{
    if (!attributes) {
        Log::Error("CheckDeckLinkAttribute called without IDeckLinkProfileAttributes");
        return false;
    }

    bool flagRes;
    auto result = attributes->GetFlag(attrID, &flagRes);
    if (FAILED(result)) {
        return false;
    }

    return flagRes;
}

bool
UpdateDeckLinkConfigurationFlag(const ScopedDeckLinkPointer<IDeckLinkConfiguration> &configuration,
                                BMDDeckLinkConfigurationID cfgID, bool val)
{
    if (!configuration) {
        Log::Error("UpdateDeckLinkConfigurationFlag called without IDeckLinkConfiguration");
        return false;
    }

    auto result = configuration->SetFlag(cfgID, val);
    if (FAILED(result)) {
        Log::Error("Unable to set flag {}", cfgID);
        return false;
    }

    return true;
}

bool
UpdateDeckLinkConfigurationInt(const ScopedDeckLinkPointer<IDeckLinkConfiguration> &configuration,
                               BMDDeckLinkConfigurationID cfgID, int val)
{
    if (!configuration) {
        Log::Error("UpdateDeckLinkConfigurationInt called without IDeckLinkConfiguration");
        return false;
    }

    auto result = configuration->SetInt(cfgID, val);
    if (FAILED(result)) {
        Log::Error("Unable to set int {}", cfgID);
        return false;
    }

    return true;
}
