#pragma once

#include "DeckLinkAPI.h"
#include "debug/log.hpp"
#include "decklink_platform.h"

#include <string>
#include <vector>

struct DeckLinkDisplayMode {
    std::string name;
    BMDDisplayMode mode;
    long width, height;
    int64_t frameDuration;
    int64_t timeScale;
};

/**
 * Helper class that wraps reference counted DeckLink interfaces to manage their lifetime
 *
 * Usage:
 * From a pointer where we already own a reference:
 *  - auto wrapped = ScopedDeckLinkPointer<IDeckLinkClass>(new IDeckLinkClass());
 *
 * From a pointer where we don't own a reference:
 *  - auto wrapped = ScopedDeckLinkPointer<IDeckLinkClass>::TakeOwnership(tempPtr);
 */
template <typename DeckLinkClass>
class ScopedDeckLinkPointer
{
public:
    ScopedDeckLinkPointer()
        : scopedPointer(nullptr)
    {
    }

    /**
     * Create from a class instance where we already own a reference
     */
    explicit ScopedDeckLinkPointer(DeckLinkClass *_ptr)
        : scopedPointer(_ptr)
    {
    }

    /**
     * Copy constructor
     */
    ScopedDeckLinkPointer(const ScopedDeckLinkPointer &o)
        : scopedPointer(o.scopedPointer)
    {
        if (this->scopedPointer) {
            this->scopedPointer->AddRef();
        }
    }

    /**
     * Move constructor
     */
    ScopedDeckLinkPointer(ScopedDeckLinkPointer &&o)
        : scopedPointer(o.scopedPointer)
    {
        o.scopedPointer = nullptr;
    }

    /**
     * Destructor
     * Release underlying reference counted pointer
     */
    ~ScopedDeckLinkPointer()
    {
        if (this->scopedPointer) {
            this->scopedPointer->Release();
        }
    }

    /**
     * Copy constructor
     */
    ScopedDeckLinkPointer &
    operator=(const ScopedDeckLinkPointer &o)
    {
        auto copy = ScopedDeckLinkPointer::TakeOwnership(o.scopedPointer);
        return (*this = std::move(copy));
    }

    /**
     * Move assignment operator
     */
    ScopedDeckLinkPointer &
    operator=(ScopedDeckLinkPointer &&o)
    {
        if (this->scopedPointer) {
            // Release pointer already held
            this->scopedPointer->Release();
        }

        // Steal pointer form mover
        this->scopedPointer = o.scopedPointer;
        o.scopedPointer     = nullptr;

        return *this;
    };

    /**
     * Copy assignment operator from raw pointer where we already own a reference
     */
    ScopedDeckLinkPointer &
    operator=(DeckLinkClass *o)
    {
        if (this->scopedPointer) {
            this->scopedPointer->Release();
        }

        this->scopedPointer = o;

        return *this;
    };

    /**
     * Dereference operator
     */
    operator DeckLinkClass *() const
    {
        return this->scopedPointer;
    }

    /**
     * Arrow operator exposes underlying pointer
     */
    DeckLinkClass *
    operator->() const
    {
        return this->scopedPointer;
    }

    /**
     * == operator between wrapped pointer and raw pointer
     */
    bool
    operator==(DeckLinkClass *o) const
    {
        return this->scopedPointer == o;
    }

    /**
     * != operator between wrapped pointer and raw pointer
     */
    bool
    operator!=(DeckLinkClass *o) const
    {
        return this->scopedPointer != o;
    }

    /**
     * == operator between two wrapped pointers
     */
    bool
    operator==(const ScopedDeckLinkPointer &o) const
    {
        return this->scopedPointer == o.scopedPointer;
    }

    /**
     * != operator between two wrapped pointers
     */
    bool
    operator!=(const ScopedDeckLinkPointer &o) const
    {
        return this->scopedPointer != o.scopedPointer;
    }

    /**
     * Bool operator (pointer != nullptr)
     */
    operator bool() const
    {
        return this->scopedPointer != nullptr;
    }

    /**
     * Get reference to underlying pointer
     */
    DeckLinkClass *&
    GetPtrRef()
    {
        return this->scopedPointer;
    }

    /**
     * Create a new ScopedDeckLinkPointer instance and take a reference to the underlying pointer
     */
    static ScopedDeckLinkPointer
    TakeOwnership(DeckLinkClass *ptr)
    {
        if (ptr != nullptr) {
            ptr->AddRef();
        }

        return ScopedDeckLinkPointer(ptr);
    }

    /**
     * Wrapper for the underlying QueryInterface function that return a new ScopedDeckLinkPointer of the target type
     *
     * Usage:
     *  - auto deckLinkOutput = deckLink.QueryInterface<IDeckLinkOutput>(IID_IDeckLinkOutput);
     *
     * Returns nullptr if underlying pointer is nullptr, or if the query failed
     */
    template <typename Target>
    ScopedDeckLinkPointer<Target>
    QueryInterface(const REFIID &ref) const
    {
        ScopedDeckLinkPointer<Target> iface;

        if (!this->scopedPointer) {
            return iface;
        }

        auto res = this->scopedPointer->QueryInterface(ref, (void **)&iface.GetPtrRef());

        if (FAILED(res)) {
            Vape::Log::Error("Failed to query {} from interface {}", typeid(Target).name(),
                             typeid(DeckLinkClass).name());
        }

        return iface;
    }

private:
    /**
     * The underlying raw pointer
     */
    DeckLinkClass *scopedPointer;
};

std::string GetDecklinkDriverVersion();

std::string GetDeckLinkName(const ScopedDeckLinkPointer<IDeckLink> &deckLink);

std::vector<DeckLinkDisplayMode> GetDisplayModes(
    const ScopedDeckLinkPointer<IDeckLinkOutput> &deckLinkOutput);

bool CheckDeckLinkAttribute(const ScopedDeckLinkPointer<IDeckLinkProfileAttributes> &attributes,
                            BMDDeckLinkAttributeID attrID);

bool UpdateDeckLinkConfigurationFlag(
    const ScopedDeckLinkPointer<IDeckLinkConfiguration> &configuration,
    BMDDeckLinkConfigurationID cfgID, bool val);
bool UpdateDeckLinkConfigurationInt(
    const ScopedDeckLinkPointer<IDeckLinkConfiguration> &configuration,
    BMDDeckLinkConfigurationID cfgID, int val);
