#pragma once
#include "./decklink-base.hpp"

#include <atomic>

/**
 * Permanent resolved IUnknown IID, to avoid multiple warnings about C99 style
 * struct initialization
 */
extern const CFUUIDBytes IID_IUnknown_Resolved;

/**
 * Implementation of the IUnknown interface expected by the DeckLink driver
 * Mimics the Windows COM interface
 * Assumes an IID definition of the same name as the class
 *
 * Class using this implementation should be created with 'new ClassName()' and
 * wrapped in a ScopedDeckLinkPointer
 * Ref counts starts at 1
 *
 * Usage:
 * Put IUNKNOWN_IMPL(ClassName) in a public section of the class declaration
 */
#define IUNKNOWN_IMPL(x)                                                 \
private:                                                                 \
    std::atomic<ULONG> m_refCount = 1;                                   \
                                                                         \
public:                                                                  \
    HRESULT QueryInterface(REFIID iid, LPVOID *ppv) override             \
    {                                                                    \
        HRESULT result = E_NOINTERFACE;                                  \
                                                                         \
        if (ppv == NULL)                                                 \
            return E_INVALIDARG;                                         \
                                                                         \
        *ppv = NULL;                                                     \
                                                                         \
        if (memcmp(&iid, &IID_IUnknown_Resolved, sizeof(REFIID)) == 0) { \
            *ppv = this;                                                 \
            AddRef();                                                    \
            result = S_OK;                                               \
        } else if (memcmp(&iid, &IID_##x, sizeof(REFIID)) == 0) {        \
            *ppv = static_cast<x *>(this);                               \
            AddRef();                                                    \
            result = S_OK;                                               \
        }                                                                \
                                                                         \
        return result;                                                   \
    }                                                                    \
                                                                         \
    ULONG                                                                \
    AddRef(void) override                                                \
    {                                                                    \
        ULONG newRefValue = ++m_refCount;                                \
        return newRefValue;                                              \
    }                                                                    \
                                                                         \
    ULONG                                                                \
    Release(void) override                                               \
    {                                                                    \
        ULONG newRefValue = --m_refCount;                                \
                                                                         \
        if (newRefValue == 0) {                                          \
            delete this;                                                 \
        }                                                                \
                                                                         \
        return newRefValue;                                              \
    }
