#include "decklink-output.hpp"

#include "../config.hpp"
#include "../image/placeholder-frame.hpp"
#include "../settings/consumer.hpp"
#include "../settings/decklink.hpp"
#include "./decklink-base.hpp"
#include "debug/log.hpp"
#include "fundamentals/helpers.hpp"
#include "thread-helpers/thread-helpers.hpp"

#include <cassert>
#include <iostream>

using namespace std;

namespace Vape {
constexpr flicks audioSampleDuration{flicks::period::den / bmdAudioSampleRate48kHz};
constexpr int frameBufferCount         = 4;
constexpr int internalBufferCount      = 4;
static thread_local bool threadNameSet = false;

static const string h("DeckLinkOutput");
static const int genlockEmitInterval = 30;

static void
SetThreadNameOnce(const char *name)
{
    if (threadNameSet) {
        return;
    }

    RenameThread(name);
    threadNameSet = true;
}

DeckLinkOutput::DeckLinkOutput(const ConsumerSettings &s)
    : allocator(new AVFrameMemoryAllocator())
{
    HRESULT res;
    const auto &deviceName      = s.outputDeviceString;
    const auto &displayModeName = s.displayModeString;
    this->device                = DeckLinkScanner::instance().GetByName(deviceName);

    auto &output = this->device.output;
    if (!output) {
        return;
    }

    auto &modes = this->device.modes;
    auto it     = find_if(modes.begin(), modes.end(),
                      [displayModeName](auto &mode) { return mode.name == displayModeName; });

    if (it != modes.end()) {
        this->displayMode = *it;
    } else {
        return;
    }

    output->SetVideoOutputFrameMemoryAllocator(this->allocator);
    output->SetScheduledFrameCompletionCallback(this);
    output->SetAudioCallback(this);

    this->configuration = output.QueryInterface<IDeckLinkConfiguration>(IID_IDeckLinkConfiguration);
    if (!this->configuration) {
        Log::TError(h, "Failed to get configuration");
        return;
    }

    this->placeholderFrame = CreatePlaceholderFrame(GetPlaceholderType(s.placeholderTypeString));
    if (!this->placeholderFrame) {
        Log::TError(h, "Failed to create placeholder frame");
        return;
    }

    res = output->EnableVideoOutput(this->displayMode->mode, bmdVideoOutputFlagDefault);
    if (FAILED(res)) {
        Log::TError(h, "Failed to enable video output");
        return;
    }

    res = output->EnableAudioOutput(bmdAudioSampleRate48kHz, bmdAudioSampleType16bitInteger, 2,
                                    bmdAudioOutputStreamContinuous);
    if (FAILED(res)) {
        Log::TError(h, "Failed to enable audio output");
        return;
    }

    auto frameDuration = this->displayMode->frameDuration;
    auto timeScale     = this->displayMode->timeScale;

    for (int i = 0; i < internalBufferCount; ++i) {
        res = output->ScheduleVideoFrame(this->placeholderFrame, this->videoPTS, frameDuration,
                                         timeScale);
        this->videoPTS += frameDuration;

        if (FAILED(res)) {
            Log::TError(h, "Failed to post initial buffer");
            return;
        }
    }

    res = output->BeginAudioPreroll();
    if (FAILED(res)) {
        Log::TError(h, "Failed to begin audio preroll");
        return;
    }

    res = output->StartScheduledPlayback(0, timeScale, 1.0);
    if (FAILED(res)) {
        Log::TError(h, "Failed to start scheduled playback");
        return;
    }

    this->valid = true;
}

DeckLinkOutput::~DeckLinkOutput()
{
    auto &output = this->device.output;
    {
        unique_lock lock(this->framesQueueMutex);
        this->stoppingOutput = true;
    }

    if (!output) {
        return;
    }

    // Call stop to flush the frame queues
    this->Stop();

    output->SetVideoOutputFrameMemoryAllocator(nullptr);
    output->SetScheduledFrameCompletionCallback(nullptr);
    output->SetAudioCallback(nullptr);

    output->StopScheduledPlayback(0, nullptr, 0);
    output->DisableAudioOutput();
    output->DisableVideoOutput();

    int itrCount              = 0;
    uint32_t framesBuffered   = 0;
    uint32_t samplesScheduled = 0;

    while (itrCount++ < 100 &&  //
           SUCCEEDED(output->GetBufferedVideoFrameCount(&framesBuffered)) &&
           SUCCEEDED(output->GetBufferedAudioSampleFrameCount(&samplesScheduled)) &&
           (framesBuffered > 0 || samplesScheduled > 0 || this->refCount > 0)) {
        this_thread::sleep_for(10ms);
    }

    if ((framesBuffered > 0 || samplesScheduled > 0 || this->refCount > 0)) {
        Log::TError(
            h, "Destructor took too long to complete, DeckLink might be left in an invalid state");
    }
}

bool
DeckLinkOutput::Valid()
{
    auto &output = this->device.output;
    return output && this->valid;
}

// IDeckLinkVideoOutputCallback
HRESULT
DeckLinkOutput::ScheduledFrameCompleted(
    /* in */ IDeckLinkVideoFrame * /*completedFrame*/,
    /* in */ BMDOutputFrameCompletionResult result)
{
    SetThreadNameOnce("ScheduledFrameCompleted");

    auto &output       = this->device.output;
    auto frameDuration = this->displayMode->frameDuration;
    auto timeScale     = this->displayMode->timeScale;

    switch (result) {
        case bmdOutputFrameCompleted:
            // If this is the genlockEmitInterval:th frame, emit genlock status
            if (++this->genlockStatusCounter == genlockEmitInterval) {
                this->genlockStatusCounter = 0;
                this->EmitGenlockStatus();
            }
            break;
        case bmdOutputFrameDisplayedLate:
            Log::TDebug(h, "Frame late, bumping output time");
            this->videoPTS += frameDuration;

            break;
        case bmdOutputFrameDropped:
            Log::TDebug(h, "Frame dropped");
            break;
        case bmdOutputFrameFlushed:
            Log::TDebug(h, "Frame flushed");
        default:
            break;
    }

    auto frame = this->GetNextVideoFrame();
    if (!frame) {
        // Output is stopping
        return S_OK;
    }

    HRESULT res = output->ScheduleVideoFrame(frame, this->videoPTS, frameDuration, timeScale);
    this->videoPTS += frameDuration;

    switch (res) {
        case E_FAIL:
            Log::Debug("ScheduledFrameCompleted E_FAIL");
            break;
        case E_INVALIDARG:
            Log::Debug("ScheduledFrameCompleted E_INVALIDARG");
            break;
        case E_ACCESSDENIED:
            Log::Debug("ScheduledFrameCompleted E_ACCESSDENIED");
            break;
        case E_OUTOFMEMORY:
            Log::Debug("ScheduledFrameCompleted E_OUTOFMEMORY");
            break;
        default:
            break;
    }

    return S_OK;
}

HRESULT
DeckLinkOutput::ScheduledPlaybackHasStopped(void)
{
    SetThreadNameOnce("ScheduledPlaybackHasStopped");

    Log::TDebug(h, "Playback stopped");
    this->stopSignal.Emit(StopReason::ConsumerStoppedPlayback);

    return S_OK;
}

HRESULT
DeckLinkOutput::RenderAudioSamples(bool preroll)
{
    /**
     * This method is called at 50hz during playback, regardless of frame rate
     */

    SetThreadNameOnce("RenderAudioSamples");

    if (this->stoppingOutput) {
        return S_OK;
    }

    auto &output = this->device.output;

    HRESULT res;
    uint32_t samplesWritten;
    uint32_t scheduledSamples;

    res = output->GetBufferedAudioSampleFrameCount(&scheduledSamples);
    if (FAILED(res)) {
        return S_OK;
    }

    // Target is 4 times the frame duration of the current display mode, to match up with the scheduled frames
    auto targetDuration = GetDisplayModeFrameDuration(*this->displayMode) * internalBufferCount;

    int64_t targetSampleCount = targetDuration.count() / audioSampleDuration.count();
    targetSampleCount -= scheduledSamples;

    while (targetSampleCount > 0) {
        auto samples = this->GetNextAudioFrame(targetSampleCount);
        if (samples.empty()) {
            // Output is stopping
            return S_OK;
        }

        auto sampleCount = samples.size() / 2;  // Interleaved stereo

        res =
            output->ScheduleAudioSamples(samples.data(), samples.size() / 2, 0, 0, &samplesWritten);

        targetSampleCount -= sampleCount;

        if (FAILED(res)) {
            Log::TError(h, "Failed to schedule audio samples");
            break;
        }
    }

    if (preroll) {
        output->EndAudioPreroll();
    }

    switch (res) {
        case E_FAIL:
            Log::TDebug(h, "RenderAudioSamples E_FAIL");
            break;
        case E_INVALIDARG:
            Log::TDebug(h, "RenderAudioSamples E_INVALIDARG");
            break;
        case E_ACCESSDENIED:
            Log::TDebug(h, "RenderAudioSamples E_ACCESSDENIED");
            break;
        case E_OUTOFMEMORY:
            Log::TDebug(h, "RenderAudioSamples E_OUTOFMEMORY");
            break;
        default:
            break;
    }

    return S_OK;
}

// Consumer
bool
DeckLinkOutput::ScheduleVideoFrame(AVFrame *frame, flicks displayTime)
{
    auto width   = this->displayMode->width;
    auto height  = this->displayMode->height;
    int lineSize = width * 2;
    int size     = lineSize * height;

    if (frame == nullptr) {
        Log::TError(h, "ScheduleVideoFrame called with nullptr frame");
        return false;
    }

    if (frame->width != width || frame->height != height) {
        Log::TError(h, "ScheduleVideoFrame called with frame of invalid size: {}x{}", frame->width,
                    frame->height);
        return false;
    }

    if (frame->linesize[0] != lineSize) {
        Log::TError(h, "ScheduleVideoFrame called with frame of invalid line size: {}",
                    frame->linesize[0]);
        return false;
    }

    if (!frame->data[0] || !frame->buf[0] || frame->buf[0]->size < size) {
        Log::TError(
            h, "ScheduleVideoFrame called with no buffer or buffer that is smaller than expected");
        return false;
    }

    unique_lock lock(this->framesQueueMutex);
    this->videoFrames.emplace(av_frame_clone(frame), displayTime);

    return true;
}

bool
DeckLinkOutput::ScheduleAudioFrame(AVFrame *frame, flicks displayTime)
{
    if (!frame) {
        Log::TError(h, "ScheduleAudioFrame called with nullptr frame");
        return false;
    }

    if (frame->channels != 2 || frame->channel_layout != AV_CH_LAYOUT_STEREO) {
        Log::TError(h, "ScheduleAudioFrame called with {} channels", frame->channels);
        return false;
    }

    if (frame->format != AV_SAMPLE_FMT_S16) {
        Log::TError(h, "ScheduleAudioFrame called with format {}",
                    av_get_sample_fmt_name((AVSampleFormat)frame->format));
        return false;
    }

    unique_lock lock(this->framesQueueMutex);
    this->audioSamplesScheduled += frame->nb_samples;
    this->audioFrames.emplace(av_frame_clone(frame), displayTime);

    return true;
}

flicks
DeckLinkOutput::VideoBufferLength()
{
    unique_lock lock(this->framesQueueMutex);
    return GetDisplayModeFrameDuration(*this->displayMode) * this->videoFrames.size();
}

flicks
DeckLinkOutput::AudioBufferLength()
{
    unique_lock lock(this->framesQueueMutex);
    return audioSampleDuration * this->audioSamplesScheduled;
}

flicks
DeckLinkOutput::TargetVideoBufferLength()
{
    return GetDisplayModeFrameDuration(*this->displayMode) * frameBufferCount;
}

flicks
DeckLinkOutput::TargetAudioBufferLength()
{
    return this->TargetVideoBufferLength();
}

bool
DeckLinkOutput::Start()
{
    unique_lock lock(this->framesQueueMutex);

    if (this->started) {
        return false;
    }

    this->started = true;
    return true;
}

bool
DeckLinkOutput::Stop()
{
    unique_lock lock(this->framesQueueMutex);

    auto flush = [](auto &queue) {
        while (!queue.empty()) {
            auto [frame, displayTime] = queue.front();
            (void)displayTime;

            queue.pop();

            av_frame_free(&frame);
        }
    };

    flush(this->videoFrames);
    flush(this->audioFrames);
    this->audioSamplesScheduled   = 0;
    this->lastScheduledVideoFrame = nullptr;

    this->started = false;

    return true;
}

bool
DeckLinkOutput::Started()
{
    unique_lock lock(this->framesQueueMutex);
    return this->started;
}

flicks
DeckLinkOutput::VideoPlayhead()
{
    unique_lock lock(this->framesQueueMutex);
    return this->playhead;
}

flicks
DeckLinkOutput::AudioPlayhead()
{
    return this->VideoPlayhead();
}

IVideoOutputConsumer::VideoFormat
DeckLinkOutput::GetVideoFormat()
{
    VideoFormat format;

    format.width         = this->displayMode->width;
    format.height        = this->displayMode->height;
    format.format        = AV_PIX_FMT_UYVY422;
    format.frameDuration = GetDisplayModeFrameDuration(*this->displayMode);
    format.scaled        = true;
    format.colorSpace    = AVCOL_SPC_BT709;
    format.colorRange    = AVCOL_RANGE_MPEG;

    return format;
}

IVideoOutputConsumer::AudioFormat
DeckLinkOutput::GetAudioFormat()
{
    AudioFormat format;

    format.format      = AV_SAMPLE_FMT_S16;
    format.layout      = AV_CH_LAYOUT_STEREO;
    format.sample_rate = bmdAudioSampleRate48kHz;

    return format;
}

DisplayModeNames
DeckLinkOutput::GetDisplayModeNames()
{
    auto &displayModes = this->device.modes;

    DisplayModeNames displayModeNames;
    displayModeNames.reserve(displayModes.size());

    for (auto &mode : displayModes) {
        // Use name for both key and value in list
        displayModeNames.emplace_back(mode.name, mode.name);
    }

    return displayModeNames;
}

bool
DeckLinkOutput::UpdateSettings(const DeckLinkSettings &s)
{
    auto &output = this->device.output;

    Log::TDebug(h, "UpdateSettings called");
    bool success = true;

    auto &conf = this->configuration;
    const auto attr =
        output.QueryInterface<IDeckLinkProfileAttributes>(IID_IDeckLinkProfileAttributes);

    if (CheckDeckLinkAttribute(attr, BMDDeckLinkSupportsFullFrameReferenceInputTimingOffset)) {
        if (UpdateDeckLinkConfigurationInt(conf, bmdDeckLinkConfigReferenceInputTimingOffset,
                                           s.genlockOffset)) {
            Log::TDebug(h, "Setting genlock offset to {}", s.genlockOffset);
        } else {
            Log::TError(h, "Failed to set timing offset");
            success = false;
        }
    } else {
        Log::TWarn(h, "Device does not support full frame reference timing offset");
        success = false;
    }

    if (CheckDeckLinkAttribute(attr, BMDDeckLinkSupportsSMPTELevelAOutput)) {
        if (UpdateDeckLinkConfigurationFlag(conf, bmdDeckLinkConfigSMPTELevelAOutput,
                                            s.sdi3GMode)) {
            Log::TDebug(h, "Setting 3G-SDI mode to {}", s.sdi3GMode);
        } else {
            Log::TError(h, "Failed to set 3G Level A");
            success = false;
        }
    } else {
        Log::TWarn(h, "Device does not support 3G Level A");
        success = false;
    }

    return success;
}

void
DeckLinkOutput::EmitGenlockStatus()
{
    auto &output = this->device.output;
    BMDReferenceStatus status;
    DeckLinkGenlockStatus result;

    auto res = output->GetReferenceStatus(&status);

    if (SUCCEEDED(res)) {
        if (status & bmdReferenceNotSupportedByHardware) {
            result = DeckLinkGenlockStatus::NotSupported;
        } else if (status & bmdReferenceLocked) {
            result = DeckLinkGenlockStatus::Locked;
        } else {
            result = DeckLinkGenlockStatus::NotLocked;
        }
    } else {
        Log::TError(h, "GetReferenceStatus call failed");
        result = DeckLinkGenlockStatus::NotSupported;
    }

    this->genlockStatusSignal.Emit(result);
}

ScopedDeckLinkPointer<IDeckLinkMutableVideoFrame>
DeckLinkOutput::CreatePlaceholderFrame(PlaceholderType type)
{
    Log::TDebug(h, "Creating placeholder frame");
    auto &output     = this->device.output;
    int width        = this->displayMode->width;
    int height       = this->displayMode->height;
    int bgraLineSize = width * 4;
    int yuvLineSize  = width * 2;

    ScopedDeckLinkPointer converter(CreateVideoConversionInstance());
    ScopedDeckLinkPointer<IDeckLinkMutableVideoFrame> bgrFrame, yuvFrame;

    auto res = output->CreateVideoFrame(width, height, bgraLineSize, bmdFormat8BitBGRA,
                                        bmdFrameFlagDefault, &bgrFrame.GetPtrRef());

    if (FAILED(res)) {
        Log::TError(h, "Failed to create rgb placeholder frame");
        return {};
    }

    res = output->CreateVideoFrame(width, height, yuvLineSize, bmdFormat8BitYUV,
                                   bmdFrameFlagDefault, &yuvFrame.GetPtrRef());

    if (FAILED(res)) {
        Log::TError(h, "Failed to create yuv placeholder frame");
        return {};
    }

    uint32_t *data;
    res = bgrFrame->GetBytes((void **)&data);
    if (FAILED(res)) {
        Log::TError(h, "Failed to get placeholder frame data pointer");
        return {};
    }

    Log::TDebug(h, "Updating placeholder frame data");
    CreatePlaceholderFrameBGRA(data, width, height, type);

    /**
     * Convert the frame to yuv, that way it doesn't get converted in the driver every time it's displayed
     */

    if (!converter) {
        Log::TError(h, "Failed to create frame converter");
        return bgrFrame;
    }

    res = converter->ConvertFrame(bgrFrame, yuvFrame);

    if (FAILED(res)) {
        Log::TError(h, "Failed to convert frame");
        return bgrFrame;
    }

    return yuvFrame;
}

ScopedDeckLinkPointer<IDeckLinkMutableVideoFrame>
DeckLinkOutput::GetNextVideoFrame()
{
    unique_lock lock(this->framesQueueMutex);
    if (this->stoppingOutput) {
        // Output is stopping, don't push new frames
        return {};
    }

    auto &output = this->device.output;

    ScopedDeckLinkPointer<IDeckLinkMutableVideoFrame> frame;
    HRESULT res;

    auto width  = this->displayMode->width;
    auto height = this->displayMode->height;

    if (this->started) {
        if (!this->videoFrames.empty()) {
            // There are frames in the queue

            // Pop the frame of the queue
            auto [avframe, displayTime] = this->videoFrames.front();
            this->videoFrames.pop();

            auto lineSize = avframe->linesize[0];

            // Push the frame to the allocator
            this->allocator->PushAVFrame(avframe);
            av_frame_free(&avframe);

            // Create the frame from the AVFrame just pushed to the allocator
            res = output->CreateVideoFrame(width, height, lineSize, bmdFormat8BitYUV,
                                           bmdFrameFlagDefault, &frame.GetPtrRef());
            if (FAILED(res)) {
                this->stopSignal.Emit(StopReason::ConsumerError);
                Log::TError(h, "Failed to create DeckLink frame from AVFrame");
                return {};
            }

            // Store last scheduled frame for potential reuse
            this->lastScheduledVideoFrame = frame;

            // Update playhead
            this->playhead = displayTime;

        } else {
            // Frame queue is empty, reuse last scheduled frame
            frame = this->lastScheduledVideoFrame;
        }
    }

    if (!frame) {
        frame = this->placeholderFrame;
    }

    return frame;
}

std::vector<int16_t>
DeckLinkOutput::GetNextAudioFrame(size_t targetSampleCount)
{
    unique_lock lock(this->framesQueueMutex);
    std::vector<int16_t> res;

    if (!this->stoppingOutput) {
        if (this->audioFrames.empty()) {
            res.insert(res.end(), targetSampleCount * 2, 0);
        } else {
            // Pop frame from queue
            auto [frame, displayTime] = this->audioFrames.front();
            this->audioFrames.pop();

            auto data        = reinterpret_cast<int16_t *>(frame->data[0]);
            auto sampleCount = frame->nb_samples;

            res.insert(res.end(), data, data + sampleCount * 2);

            av_frame_free(&frame);

            this->audioSamplesScheduled -= sampleCount;
        }
    }

    return res;
}

}  // namespace Vape
