#pragma once

#include "../image/placeholder-type.hpp"
#include "./decklink-allocator.hpp"
#include "./decklink-base.hpp"
#include "./decklink-genlock-status.hpp"
#include "./decklink-scanner.hpp"
#include "./ivideooutputconsumer.hpp"
#include "fundamentals/signal-provider.hpp"
#include "vapour-player-core/time.hpp"

#include <atomic>
#include <optional>
#include <queue>
#include <vector>

namespace Vape {

struct ConsumerSettings;
struct DeckLinkSettings;

class DeckLinkOutput final
    : public IDeckLinkVideoOutputCallback
    , public IDeckLinkAudioOutputCallback
    , public IVideoOutputConsumer
{
public:
    DeckLinkOutput(const ConsumerSettings &s);
    virtual ~DeckLinkOutput();

    // IDeckLinkVideoOutputCallback
    HRESULT ScheduledFrameCompleted(
        /* in */ IDeckLinkVideoFrame *completedFrame,
        /* in */ BMDOutputFrameCompletionResult result) final;
    HRESULT ScheduledPlaybackHasStopped(void) final;

    // IDeckLinkAudioOutputCallback
    HRESULT RenderAudioSamples(bool preroll) final;

    // IUnknown
    virtual HRESULT STDMETHODCALLTYPE
    QueryInterface(REFIID /*iid*/, LPVOID * /*ppv*/) override
    {
        return E_NOINTERFACE;
    }
    virtual ULONG STDMETHODCALLTYPE
    AddRef(void) override
    {
        ULONG newRefValue = ++this->refCount;
        return newRefValue;
    }

    virtual ULONG STDMETHODCALLTYPE
    Release(void) override
    {
        ULONG newRefValue = --this->refCount;
        return newRefValue;
    }

    // Consumer
    virtual bool ScheduleVideoFrame(AVFrame *frame, flicks displayTime) override;
    virtual bool ScheduleAudioFrame(AVFrame *frame, flicks displayTime) override;

    virtual bool Valid() override;
    virtual bool Start() override;
    virtual bool Stop() override;
    virtual bool Started() override;
    virtual flicks VideoPlayhead() override;
    virtual flicks AudioPlayhead() override;
    virtual flicks VideoBufferLength() override;
    virtual flicks AudioBufferLength() override;
    virtual flicks TargetVideoBufferLength() override;
    virtual flicks TargetAudioBufferLength() override;

    virtual VideoFormat GetVideoFormat() override;
    virtual AudioFormat GetAudioFormat() override;

    virtual DisplayModeNames GetDisplayModeNames() override;

    bool UpdateSettings(const DeckLinkSettings &);

    Signal::Provider<DeckLinkGenlockStatus> genlockStatusSignal;

private:
    /**
     * Tracks hos many DeckLink classes holds references to this class
     * Used in the destructor to wait for DeckLink stop
     */
    std::atomic<ULONG> refCount{0};

    /**
     * Set to true if the constructor runs without error
     */
    bool valid{false};

    /**
     * Check genlock status and emit the genlock signal
     */
    void EmitGenlockStatus();

    /**
     * Create a placeholder frame to be used when playback is not active
     * @param type The type of frame to create
     * @return The frame wrapped in a ScopedDeckLinkPointer, or a wrapped nullptr if the creation fails
     */
    ScopedDeckLinkPointer<IDeckLinkMutableVideoFrame> CreatePlaceholderFrame(PlaceholderType type);

    /**
     * Helper function for ScheduledFrameCompleted
     * @return Next frame from queue, reuse last frame, placeholder frame, or nullptr if device is stopping
     */
    ScopedDeckLinkPointer<IDeckLinkMutableVideoFrame> GetNextVideoFrame();

    /**
     * Helper function for RenderAudioSamples
     * @return Contents of next frame in queue, silence of exactly targetSampleCount, or empty vector if device is stopping
     */
    std::vector<int16_t> GetNextAudioFrame(size_t targetSampleCount);

    /**
     * Copy of the device description containing the wrapped device pointer and an array of valid display modes
     */
    DeckLinkScanner::CachedDevice device;

    /**
     * The configuration object queried from the device
     */
    ScopedDeckLinkPointer<IDeckLinkConfiguration> configuration;

    /**
     * The custom AVFrame allocator
     */
    ScopedDeckLinkPointer<AVFrameMemoryAllocator> allocator;

    /**
     * The placeholder frame used when the playback is not active
     */
    ScopedDeckLinkPointer<IDeckLinkMutableVideoFrame> placeholderFrame;

    /**
     * The active display mode
     */
    std::optional<DeckLinkDisplayMode> displayMode;

    /**
     * -----------------------------------------------------------
     * Mutex for all operation on the class members declared below
     */
    std::mutex framesQueueMutex;

    /**
     * The queues for video and audio frames that the decoder freeds to and the internal
     * callbacks pull from
     */
    std::queue<std::pair<AVFrame *, flicks>> videoFrames, audioFrames;

    /**
     * The last frame the internal callback has pulled from the queue
     */
    ScopedDeckLinkPointer<IDeckLinkMutableVideoFrame> lastScheduledVideoFrame;

    /**
     * The PTS of the next video frame to be scheduled
     */
    BMDTimeValue videoPTS{0};

    /**
     * Number of audio samples in the audioFrames queue
     */
    uint64_t audioSamplesScheduled{0};

    /**
     * The PTS of the last video frame to be pulled form the queue by the callback
     */
    flicks playhead{0};

    /**
     * The consumer has set the output to started and the internal callbacks
     * should pull data from the queues
     */
    bool started{false};

    /**
     * The destructor has set the output to stopping, so in case any of the
     * callbacks still fire, no new frames are scheduled
     */
    bool stoppingOutput{false};

    /**
     * The counter for how many frames has been scheduled since last genlock status was broadcast
     */
    int genlockStatusCounter{0};
};

}  // namespace Vape
