#include "./decklink-scanner.hpp"

#include "debug/log.hpp"

#include <DeckLinkAPIVersion.h>

namespace Vape {

/**
 * Name for logging
 */
static const std::string h{"DeckLinkScanner"};

DeckLinkScanner::Callback::Callback(DeckLinkScanner &_scanner)
    : scanner(_scanner)
{
}

HRESULT
DeckLinkScanner::Callback::DeckLinkDeviceArrived(IDeckLink *deckLinkDevice)
{
    return this->scanner.DeckLinkDeviceArrived(deckLinkDevice);
}

HRESULT
DeckLinkScanner::Callback::DeckLinkDeviceRemoved(IDeckLink *deckLinkDevice)
{
    return this->scanner.DeckLinkDeviceRemoved(deckLinkDevice);
}

DeckLinkScanner::DeckLinkScanner()
{
}

DeckLinkScanner::~DeckLinkScanner()
{
    this->Deinit();
}

void
DeckLinkScanner::SignalDeviceNames()
{
    std::vector<std::string> deviceNames;

    for (auto &device : this->devices) {
        deviceNames.push_back(device.first);
    }

    this->deviceNamesSignal.Emit(deviceNames);
}

HRESULT
DeckLinkScanner::DeckLinkDeviceArrived(IDeckLink *deckLinkDevice)
{
    std::unique_lock lock(this->deviceListMutex);

    auto deckLink     = ScopedDeckLinkPointer<IDeckLink>::TakeOwnership(deckLinkDevice);
    auto outputDevice = deckLink.QueryInterface<IDeckLinkOutput>(IID_IDeckLinkOutput);

    if (outputDevice) {
        auto name  = GetDeckLinkName(deckLink);
        auto modes = GetDisplayModes(outputDevice);

        std::string allModes;

        Log::TInfo(h, "Found DeckLink output: {}", name);

        for (auto &mode : modes) {
            if (!allModes.empty()) {
                allModes.append(", ");
            }

            allModes.append(mode.name);
        }

        Log::TDebug(h, "Modes: {}", allModes);

        // Get the ProfileManager from the output to set the duplex mode
        auto manager =
            outputDevice.QueryInterface<IDeckLinkProfileManager>(IID_IDeckLinkProfileManager);
        if (manager) {
            Log::TTrace(h, "Device {} has profile manager", name);

            ScopedDeckLinkPointer<IDeckLinkProfileIterator> profileItr;
            auto res = manager->GetProfiles(&profileItr.GetPtrRef());

            if (SUCCEEDED(res)) {
                Log::TTrace(h, "Device {} got profile iterator", name);

                // Iterate all profiles to find the one with half duplex and more than one subdevice
                IDeckLinkProfile *profilePtr;
                while (profileItr->Next(&profilePtr) == S_OK) {
                    ScopedDeckLinkPointer profile(profilePtr);

                    auto attr = profile.QueryInterface<IDeckLinkProfileAttributes>(
                        IID_IDeckLinkProfileAttributes);

                    if (!attr) {
                        continue;
                    }

                    int64_t mode;
                    res = attr->GetInt(BMDDeckLinkDuplex, &mode);
                    if (FAILED(res)) {
                        continue;
                    }

                    // Require more that one subdevice to avoid the profile bmdProfileOneSubDeviceHalfDuplex
                    int64_t subDevices;
                    res = attr->GetInt(BMDDeckLinkNumberOfSubDevices, &subDevices);
                    if (FAILED(res)) {
                        continue;
                    }

                    if (mode == bmdDuplexHalf && subDevices > 1) {
                        res = profile->SetActive();
                        if (SUCCEEDED(res)) {
                            Log::TInfo(h, "{} was successfully set to half duplex", name);
                        } else {
                            Log::TWarn(h, "{} could not be set to half duplex", name);
                        }

                        // Stop iteration
                        break;
                    }
                }
            } else {
                Log::TWarn(h, "Failed to get ProfileIterator for {}", name);
            }
        } else {
            Log::TWarn(h, "Failed to get ProfileManager for {}", name);
        }

        CachedDevice d{std::move(outputDevice), std::move(modes)};
        this->devices.emplace(std::move(name), std::move(d));
    }

    SignalDeviceNames();

    return S_OK;
}

HRESULT
DeckLinkScanner::DeckLinkDeviceRemoved(IDeckLink *deckLinkDevice)
{
    std::unique_lock lock(this->deviceListMutex);

    auto deckLink = ScopedDeckLinkPointer<IDeckLink>::TakeOwnership(deckLinkDevice);
    auto name     = GetDeckLinkName(deckLink);

    if (!name.empty()) {
        this->devices.erase(name);
        SignalDeviceNames();
    }

    return S_OK;
}

void
DeckLinkScanner::Init()
{
    Log::TInfo(h, "Initialized DeckLinkScanner (SDK version {}, Driver version {})",
               BLACKMAGIC_DECKLINK_API_VERSION_STRING, GetDecklinkDriverVersion());

    this->discovery = CreateDeckLinkDiscoveryInstance();
    this->callback  = new Callback(*this);

    if (this->discovery) {
        this->discovery->InstallDeviceNotifications(this->callback);
    }
}

void
DeckLinkScanner::Deinit()
{
    if (this->discovery) {
        this->discovery->UninstallDeviceNotifications();
        this->callback  = nullptr;
        this->discovery = nullptr;
    }

    std::unique_lock lock(this->deviceListMutex);
    this->devices.clear();
}

DeckLinkScanner::CachedDevice
DeckLinkScanner::GetByName(const std::string &name)
{
    CachedDevice device;
    Log::TTrace(h, "Get device: {}", name);

    std::unique_lock lock(this->deviceListMutex);
    auto find = this->devices.find(name);
    if (find != this->devices.end()) {
        device = find->second;
    }

    return device;
}

DeckLinkScanner &
DeckLinkScanner::instance()
{
    static DeckLinkScanner scanner;
    return scanner;
}

}  // namespace Vape
