#pragma once
#include "./decklink-base.hpp"
#include "./decklink-iunknown.hpp"
#include "fundamentals/signal-provider.hpp"

#include <map>
#include <mutex>
#include <string>

namespace Vape {

/**
 * Handles DeckLink device detection by receiving notifications from the driver
 * In practice, this will be a static list that is received on application start, but for unknown reasons
 * the iterator approach missbehaves on some server configurations.
 */
class DeckLinkScanner
{
public:
    /**
     * The struct we store for each device containing the wrapped device pointer and
     * the display modes we have queried for the device
     */
    struct CachedDevice {
        ScopedDeckLinkPointer<IDeckLinkOutput> output;
        std::vector<DeckLinkDisplayMode> modes;
    };

private:
    /**
     * The callback class that implements the driver callback interface
     */
    class Callback : public IDeckLinkDeviceNotificationCallback
    {
        DeckLinkScanner &scanner;

    public:
        Callback(DeckLinkScanner &_scanner);

        /**
         * Reference counting implementation macro for IUnknown
         */
        IUNKNOWN_IMPL(IDeckLinkDeviceNotificationCallback);

        /**
         * Fulfills IDeckLinkDeviceNotificationCallback
         */
        HRESULT DeckLinkDeviceArrived(IDeckLink *deckLinkDevice) override;
        HRESULT DeckLinkDeviceRemoved(IDeckLink *deckLinkDevice) override;
    };

    /**
     * The map of detected devices with
     * name -> { device pointer, display modes }
     */
    std::map<std::string, CachedDevice> devices;
    std::mutex deviceListMutex;

    /**
     * Instance of IDeckLinkDiscovery created by Init()
     */
    ScopedDeckLinkPointer<IDeckLinkDiscovery> discovery;

    /**
     * Instance of DeckLinkScanner::Callback created by Init()
     */
    ScopedDeckLinkPointer<Callback> callback;

    /**
     * Create vector of device names and signal it
     */
    void SignalDeviceNames();

public:
    DeckLinkScanner();
    ~DeckLinkScanner();

    /**
     * Create IDeckLinkDiscovery and Callback instance, and install callback
     */
    void Init();

    /**
     *  Uninstall callback, clear instances and map
     */
    void Deinit();

    /**
     * Get cached device and display mode list by name
     * If no device is found, the device will be nullptr and the mode list empty
     */
    CachedDevice GetByName(const std::string &name);

    /**
     * Indirect call from Callback
     * Queries device for IDeckLinkOutput and display modes
     * Inserts device in map
     */
    HRESULT DeckLinkDeviceArrived(IDeckLink *deckLinkDevice);

    /**
     * Indirect call from Callback
     * Removes device from map by name
     */
    HRESULT DeckLinkDeviceRemoved(IDeckLink *deckLinkDevice);

    /**
     * Signals a vector of device names every time a notification in received from the driver
     */
    Signal::Provider<std::vector<std::string>> deviceNamesSignal;

    /**
     * Singleton for the DeckLinkScanner class
     */
    static DeckLinkScanner &instance();
};
}  // namespace Vape
