#pragma once

#include "../stop-reason.hpp"
#include "fundamentals/signal-provider.hpp"
#include "vapour-player-core/time.hpp"
#include "vapour-player-core/types/streamindex.hpp"

extern "C" {
#include <libavutil/channel_layout.h>
#include <libavutil/frame.h>
#include <libavutil/pixfmt.h>
#include <libavutil/samplefmt.h>
}

#include <utility>
#include <vector>

namespace Vape {

using DisplayModeNames = std::vector<std::pair<std::string, std::string>>;

class IVideoOutputConsumer
{
public:
    struct VideoFormat {
        AVPixelFormat format;
        uint32_t width, height;
        flicks frameDuration;
        bool scaled;
        AVColorSpace colorSpace;
        AVColorRange colorRange;
    };

    struct AudioFormat {
        AVSampleFormat format;
        int64_t sample_rate;
        int64_t layout;
    };

    virtual ~IVideoOutputConsumer() = default;

    virtual bool ScheduleVideoFrame(AVFrame *frame, flicks displayTime) = 0;
    virtual bool ScheduleAudioFrame(AVFrame *frame, flicks displayTime) = 0;
    bool ScheduleFrame(StreamIndex index, AVFrame *frame, flicks displayTime);

    virtual bool Valid()   = 0;
    virtual bool Start()   = 0;
    virtual bool Stop()    = 0;
    virtual bool Started() = 0;

    virtual flicks VideoPlayhead() = 0;
    virtual flicks AudioPlayhead() = 0;
    flicks Playhead(StreamIndex index);

    virtual flicks VideoBufferLength() = 0;
    virtual flicks AudioBufferLength() = 0;
    flicks BufferLength(StreamIndex index);

    virtual flicks TargetVideoBufferLength() = 0;
    virtual flicks TargetAudioBufferLength() = 0;
    flicks TargetBufferLength(StreamIndex index);

    virtual VideoFormat GetVideoFormat() = 0;
    virtual AudioFormat GetAudioFormat() = 0;

    virtual DisplayModeNames GetDisplayModeNames() = 0;

    Signal::Provider<StopReason> stopSignal;
};

}  // namespace Vape
