#include "./ndi-output.hpp"

#include "../config.hpp"
#include "../image/placeholder-frame.hpp"
#include "../utils/thread-affinity.hpp"
#include "debug/log.hpp"
#include "settings/consumer.hpp"

#include <functional>

namespace Vape {

static const int frameBufferCount = 5;

static const std::vector<NDIDisplayMode> displayModes = {
    {"Unscaled 29.97", 0, 0, 30000, 1001, false},  {"Unscaled 30", 0, 0, 30000, 1000, false},
    {"Unscaled 59.94", 0, 0, 60000, 1001, false},  {"Unscaled 60", 0, 0, 60000, 1000, false},
    {"720p50", 1280, 720, 50000, 1000, true},      {"720p59.94", 1280, 720, 60000, 1001, true},
    {"720p60", 1280, 720, 60000, 1000, true},      {"1080p23.98", 1920, 1080, 24000, 1001, true},
    {"1080p24", 1920, 1080, 24000, 1000, true},    {"1080p25", 1920, 1080, 25000, 1000, true},
    {"1080p29.97", 1920, 1080, 30000, 1001, true}, {"1080p30", 1920, 1080, 30000, 1000, true},
    {"1080p50", 1920, 1080, 50000, 1000, true},    {"1080p59.94", 1920, 1080, 60000, 1001, true},
    {"1080p60", 1920, 1080, 60000, 1000, true},
};

NDIVideoFrame::NDIVideoFrame(AVFrame *frame, const NDIDisplayMode &mode, flicks pts)
    : PTS(pts)
    , avFrame(av_frame_clone(frame))
    , ndiFrame(std::make_unique<NDIlib_video_frame_v2_t>())
{
    this->ndiFrame->xres                 = this->avFrame->width;
    this->ndiFrame->yres                 = this->avFrame->height;
    this->ndiFrame->FourCC               = NDIlib_FourCC_type_NV12;
    this->ndiFrame->frame_format_type    = NDIlib_frame_format_type_progressive;
    this->ndiFrame->line_stride_in_bytes = this->avFrame->linesize[0];
    this->ndiFrame->frame_rate_N         = mode.timeScale;
    this->ndiFrame->frame_rate_D         = mode.frameDuration;
    this->ndiFrame->picture_aspect_ratio = (float)this->ndiFrame->xres / this->ndiFrame->yres;
    this->ndiFrame->timecode             = NDIlib_send_timecode_synthesize;

    this->ndiFrame->p_data = nullptr;
}

NDIVideoFrame::~NDIVideoFrame()
{
    free(this->ndiFrame->p_data);
    av_frame_free(&this->avFrame);
}

const NDIlib_video_frame_v2_t *
NDIVideoFrame::GetNDIFrame()
{
    if (this->ndiFrame->p_data == nullptr) {
        size_t lumaSize   = this->avFrame->linesize[0] * this->avFrame->height;
        size_t chromaSize = this->avFrame->linesize[1] * this->avFrame->height / 2;

        this->ndiFrame->p_data = (uint8_t *)malloc(chromaSize + lumaSize);

        // Copy luma data to the start of the frame
        std::copy(this->avFrame->data[0], this->avFrame->data[0] + lumaSize,
                  this->ndiFrame->p_data);

        // Copy chroma data to the end of the frame
        std::copy(this->avFrame->data[1], this->avFrame->data[1] + chromaSize,
                  this->ndiFrame->p_data + lumaSize);
    }

    return this->ndiFrame.get();
}

NDIAudioFrame::NDIAudioFrame(AVFrame *frame, flicks pts)
    : PTS(pts)
    , avFrame(av_frame_clone(frame))
    , ndiFrame(std::make_unique<NDIlib_audio_frame_interleaved_16s_t>())
{
    this->ndiFrame->sample_rate = this->avFrame->sample_rate;
    this->ndiFrame->no_channels = this->avFrame->channels;
    this->ndiFrame->no_samples  = this->avFrame->nb_samples;

    this->ndiFrame->p_data = reinterpret_cast<short *>(this->avFrame->data[0]);
}

NDIAudioFrame::~NDIAudioFrame()
{
    av_frame_free(&this->avFrame);
}

const NDIlib_audio_frame_interleaved_16s_t *
NDIAudioFrame::GetNDIFrame()
{
    return this->ndiFrame.get();
}

NDIOutput::NDIOutput(const ConsumerSettings &s)
    : placeholderType(GetPlaceholderType(s.placeholderTypeString))
{
    const auto &displayModeName = s.displayModeString;

    auto it = std::find_if(displayModes.begin(), displayModes.end(),
                           [displayModeName](auto &mode) { return mode.name == displayModeName; });
    if (it != displayModes.end()) {
        this->displayMode = &*it;
    } else {
        this->displayMode = &displayModes.front();
    }

    if (this->displayMode != nullptr) {
        const auto &name = Config::Get().ndiOutputName;

        NDIlib_send_create_t NDI_send_create_desc;
        NDI_send_create_desc.p_ndi_name  = name.c_str();
        NDI_send_create_desc.clock_video = true;
        NDI_send_create_desc.clock_audio = true;

        this->sendInstance = NDIlib_send_create(&NDI_send_create_desc);
        if (this->sendInstance) {
            this->OutputPlaceholderFrame();
        }
    }
}

NDIOutput::~NDIOutput()
{
    if (this->sendInstance != nullptr) {
        NDIlib_send_destroy(this->sendInstance);
    }
}

bool
NDIOutput::Valid()
{
    return this->sendInstance != nullptr;
}

// Consumer
bool
NDIOutput::ScheduleVideoFrame(AVFrame *frame, flicks displayTime)
{
    auto ndiFrame = std::make_shared<NDIVideoFrame>(frame, *this->displayMode, displayTime);

    {
        std::unique_lock lock(this->videoBufferMutex);
        this->videoFramesBuffer.push_back(ndiFrame);
    }

    return true;
}

bool
NDIOutput::ScheduleAudioFrame(AVFrame *frame, flicks displayTime)
{
    auto ndiFrame = std::make_shared<NDIAudioFrame>(frame, displayTime);

    {
        std::unique_lock lock(this->audioBufferMutex);
        auto rawFrame = ndiFrame->GetNDIFrame();
        this->scheduledSamples += rawFrame->no_samples;

        this->audioFramesBuffer.push_back(ndiFrame);
    }

    return true;
}

flicks
NDIOutput::VideoBufferLength()
{
    std::shared_lock lock(this->videoBufferMutex);
    size_t scheduledFrames = this->videoFramesBuffer.size();

    return GetDisplayModeFrameDuration(*this->displayMode) * scheduledFrames;
}

flicks
NDIOutput::AudioBufferLength()
{
    std::shared_lock lock(this->audioBufferMutex);

    constexpr flicks sampleDuration{flicks::period::den / 48000};
    return sampleDuration * this->scheduledSamples;
}

flicks
NDIOutput::TargetVideoBufferLength()
{
    return GetDisplayModeFrameDuration(*this->displayMode) * frameBufferCount;
}

flicks
NDIOutput::TargetAudioBufferLength()
{
    // NOTE(Love): Returning the video buffer length as the audio target buffer is intentional
    // as this behaves better on frame drops and on the initial filling of the buffer
    return this->TargetVideoBufferLength();
}

bool
NDIOutput::Start()
{
    assert(!this->playVideoThread && !this->playAudioThread);
    this->isStarted = true;

    this->playVideoThread = std::make_unique<std::thread>(std::bind(&NDIOutput::RunVideo, this));
    this->playAudioThread = std::make_unique<std::thread>(std::bind(&NDIOutput::RunAudio, this));

    return true;
}

bool
NDIOutput::Stop()
{
    this->isStarted = false;

    if (this->playVideoThread) {
        this->playVideoThread->join();
        this->playVideoThread.reset();
    }

    if (this->playAudioThread) {
        this->playAudioThread->join();
        this->playAudioThread.reset();
    }

    {
        std::unique_lock lock(this->videoBufferMutex);
        this->videoFramesBuffer.clear();
    }

    {
        std::unique_lock lock(this->audioBufferMutex);
        this->audioFramesBuffer.clear();
        this->scheduledSamples = 0;
    }

    this->videoPlayhead = flicks{0};
    this->audioPlayhead = flicks{0};

    this->OutputPlaceholderFrame();

    return true;
}

bool
NDIOutput::Started()
{
    return this->isStarted;
}

flicks
NDIOutput::VideoPlayhead()
{
    return this->videoPlayhead;
}

flicks
NDIOutput::AudioPlayhead()
{
    return this->audioPlayhead;
}

IVideoOutputConsumer::VideoFormat
NDIOutput::GetVideoFormat()
{
    VideoFormat format;

    format.width         = this->displayMode->width;
    format.height        = this->displayMode->height;
    format.format        = AV_PIX_FMT_NV12;
    format.frameDuration = GetDisplayModeFrameDuration(*this->displayMode);
    format.scaled        = this->displayMode->scaled;
    format.colorSpace    = AVCOL_SPC_BT709;
    format.colorRange    = AVCOL_RANGE_MPEG;

    return format;
}

IVideoOutputConsumer::AudioFormat
NDIOutput::GetAudioFormat()
{
    AudioFormat format;

    format.format      = AV_SAMPLE_FMT_S16;
    format.layout      = AV_CH_LAYOUT_STEREO;
    format.sample_rate = 48000;

    return format;
}

DisplayModeNames
NDIOutput::GetDisplayModeNames()
{
    static DisplayModeNames cache = [] {
        DisplayModeNames displayModeNames;
        displayModeNames.reserve(displayModes.size());

        for (auto &mode : displayModes) {
            displayModeNames.push_back({mode.name, mode.name});
        }

        return displayModeNames;
    }();

    return cache;
}

void
NDIOutput::RunVideo()
{
    SetThreadNameAndAffinity("VMS:ndiv");

    auto GetVideoFrame = [this]() -> std::shared_ptr<NDIVideoFrame> {
        std::unique_lock lock(this->videoBufferMutex);
        if (this->videoFramesBuffer.empty()) {
            return nullptr;
        }

        auto frame = this->videoFramesBuffer.front();
        this->videoFramesBuffer.pop_front();

        return frame;
    };

    while (this->isStarted) {
        if (auto frame = GetVideoFrame()) {
            NDIlib_send_send_video_v2(this->sendInstance, frame->GetNDIFrame());
            this->videoPlayhead = frame->PTS;
        } else {
            std::this_thread::sleep_for(std::chrono::milliseconds(1));
        }
    }
}

void
NDIOutput::RunAudio()
{
    SetThreadNameAndAffinity("VMS:ndia");

    auto GetAudioFrame = [this]() -> std::shared_ptr<NDIAudioFrame> {
        std::unique_lock lock(this->audioBufferMutex);
        if (this->audioFramesBuffer.empty()) {
            return nullptr;
        }

        auto frame = this->audioFramesBuffer.front();
        this->audioFramesBuffer.pop_front();

        auto ndiFrame = frame->GetNDIFrame();
        this->scheduledSamples -= ndiFrame->no_samples;

        return frame;
    };

    while (this->isStarted) {
        if (auto frame = GetAudioFrame()) {
            auto ndiFrame = frame->GetNDIFrame();
            NDIlib_util_send_send_audio_interleaved_16s(this->sendInstance, ndiFrame);
            this->audioPlayhead = frame->PTS;
        } else {
            std::this_thread::sleep_for(std::chrono::milliseconds(1));
        }
    }
}

void
NDIOutput::OutputPlaceholderFrame()
{
    size_t width = 1280, height = 720;

    if (displayMode->scaled) {
        width  = displayMode->width;
        height = displayMode->height;
    }

    NDIlib_video_frame_v2_t frame = {};
    frame.xres                    = width;
    frame.yres                    = height;
    frame.FourCC                  = NDIlib_FourCC_type_BGRX;
    frame.frame_format_type       = NDIlib_frame_format_type_progressive;
    frame.line_stride_in_bytes    = width * 4;
    frame.frame_rate_N            = displayMode->timeScale;
    frame.frame_rate_D            = displayMode->frameDuration;
    frame.picture_aspect_ratio    = (float)width / height;
    frame.timecode                = NDIlib_send_timecode_synthesize;
    frame.p_data                  = (uint8_t *)malloc(width * height * 4);

    if (frame.p_data) {
        uint32_t *data = reinterpret_cast<uint32_t *>(frame.p_data);
        CreatePlaceholderFrameBGRA(data, width, height, this->placeholderType);

        NDIlib_send_send_video_v2(this->sendInstance, &frame);

        free(frame.p_data);
    }
}

}  // namespace Vape
