#pragma once

#include "../image/placeholder-type.hpp"
#include "./ivideooutputconsumer.hpp"

#include <Processing.NDI.Lib.h>

#include <atomic>
#include <deque>
#include <memory>
#include <shared_mutex>
#include <thread>
#include <vector>

namespace Vape {

struct ConsumerSettings;

struct NDIDisplayMode {
    std::string name;
    long width, height;
    int64_t timeScale;
    int64_t frameDuration;
    bool scaled;
};

class NDIVideoFrame
{
public:
    NDIVideoFrame(AVFrame *frame, const NDIDisplayMode &mode, flicks pts);
    ~NDIVideoFrame();

    const NDIlib_video_frame_v2_t *GetNDIFrame();
    const flicks PTS;

private:
    AVFrame *avFrame;
    std::unique_ptr<NDIlib_video_frame_v2_t> ndiFrame;
};

class NDIAudioFrame
{
public:
    NDIAudioFrame(AVFrame *frame, flicks pts);
    ~NDIAudioFrame();

    const NDIlib_audio_frame_interleaved_16s_t *GetNDIFrame();
    const flicks PTS;

private:
    AVFrame *avFrame;
    std::unique_ptr<NDIlib_audio_frame_interleaved_16s_t> ndiFrame;
};

class NDIOutput final : public IVideoOutputConsumer
{
public:
    NDIOutput(const ConsumerSettings &s);
    ~NDIOutput() final;

    // Consumer
    virtual bool ScheduleVideoFrame(AVFrame *frame, flicks displayTime) override;
    virtual bool ScheduleAudioFrame(AVFrame *frame, flicks displayTime) override;

    virtual bool Valid() override;
    virtual bool Start() override;
    virtual bool Stop() override;
    virtual bool Started() override;
    virtual flicks VideoPlayhead() override;
    virtual flicks AudioPlayhead() override;
    virtual flicks VideoBufferLength() override;
    virtual flicks AudioBufferLength() override;
    virtual flicks TargetVideoBufferLength() override;
    virtual flicks TargetAudioBufferLength() override;

    virtual VideoFormat GetVideoFormat() override;
    virtual AudioFormat GetAudioFormat() override;

    virtual DisplayModeNames GetDisplayModeNames() override;

private:
    void RunVideo();
    void RunAudio();

    void OutputPlaceholderFrame();

    std::atomic<flicks> videoPlayhead{flicks{0}};
    std::atomic<flicks> audioPlayhead{flicks{0}};

    std::unique_ptr<std::thread> playVideoThread;
    std::unique_ptr<std::thread> playAudioThread;
    std::atomic<bool> isStarted{false};

    const NDIDisplayMode *displayMode;
    const PlaceholderType placeholderType;
    NDIlib_send_instance_t sendInstance = nullptr;

    std::shared_mutex videoBufferMutex;
    std::deque<std::shared_ptr<NDIVideoFrame>> videoFramesBuffer;

    std::shared_mutex audioBufferMutex;
    std::deque<std::shared_ptr<NDIAudioFrame>> audioFramesBuffer;
    uint32_t scheduledSamples = 0;
};
}  // namespace Vape
