#include "./manager.hpp"

#include "../config.hpp"
#include "api/client.hpp"
#include "api/parse.hpp"
#include "debug/log.hpp"
#include "fundamentals/helpers.hpp"
#include "pubsub/client.hpp"

namespace Vape {

static const std::string h{"CredentialsManager"};

CredentialsManager::CredentialsManager(PubSub::Client &pubsubClient_, API::Client &apiClient_)
    : pubsubClient(pubsubClient_)
    , apiClient(apiClient_)
{
    // Avoid warning for build cases where pubsubClient is not used
    (void)this->pubsubClient;
}

std::pair<std::string, std::string>
CredentialsManager::ResolveCredentials()
{
    auto &conf = Config::Get();

    /**
     * If we don't have a previous valid whois response, make a whois request to Authenticator
     */
#if 1
    if (this->whoisData == std::nullopt) {
        auto url = fS("https://{}:443/api/authenticator/v1/whois", conf.coreHost);
        auto req = API::Client::CreateGetRequest(url);
        auto res = apiClient.ExecuteSync(std::move(req));

        auto [whois, err] = API::ParseResponse<AuthenticatorWhoIs>(res, {200});
        if (err) {
            Log::Error(err.message);
        } else {
            this->whoisData = whois;
        }
    }
#endif

#if 1
    /**
     * Make a credentals request to Authenticator
     */
    {
        auto url = fS("https://{}:443/api/authenticator/v1/service/provider-tokens", conf.coreHost);

        auto req = API::Client::CreateGetRequest(url);
        req.SetHeaders(this->pubsubClient.GetIdentityHeaders(true));

        auto res         = apiClient.ExecuteSync(std::move(req));
        auto [cred, err] = API::ParseResponse<AuthenticatorCredentials>(res, {200});

        if (err) {
            Log::TError(h, err.message);
        } else {
            return {cred.clientID, cred.accessToken};
        }
    }
#endif

#if 0
    /**
     * Query the old Authenticator
     * May be left in as long as the env variables are removed
     */
    {
        auto &userID               = conf.authenticatorUserID;
        auto &url                  = conf.authenticatorUrl;
        auto &tokenRefreshPassword = conf.authenticatorRefreshPassword;

        if (!userID.empty() && !url.empty() && !tokenRefreshPassword.empty()) {
            const std::vector<std::pair<std::string, std::string>> refreshValues{
                {"userID", userID},
                {"password", tokenRefreshPassword},
            };

            auto req = API::Client::CreateFormPostRequest(url, refreshValues);

            auto res         = apiClient.ExecuteSync(std::move(req));
            auto [cred, err] = API::ParseResponse<AuthenticatorOld>(res, {200});

            if (err) {
                Log::TError(h, err.message);
            } else {
                const auto &c = Config::Get();
                return {c.fallbackClientID, cred.accessToken};
            }
        }
    }
#endif

    /**
     * The credentials request failed, check if we have valid whois data
     */
    if (this->whoisData != std::nullopt) {
        Log::TDebug(h, "Using whois credentials");
        return {this->whoisData->clientID, ""};
    }

    /**
     * Fall back to the hard coded client ID
     */
    const auto &c = Config::Get();
    Log::TDebug(h, "Using fallback credentials");
    return {c.fallbackClientID, ""};
}

}  // namespace Vape
