#pragma once
#include "../processing/playbackrate.hpp"
#include "../stop-reason.hpp"
#include "../utils/queue-average.hpp"
#include "./sample-buffer.hpp"
#include "vapour-player-core/time.hpp"

#include <functional>

extern "C" {
#include <libavutil/frame.h>
}

namespace Vape {
constexpr flicks maxPTSSearch = toFlicks(10.f);

/**
 * The interface that all decoders inherit from
 */
class IDecoder
{
protected:
    SampleBuffer sampleBuffer;
    QueueAverage<flicks128, 300> averagePTStoIngestTimeOffset;

    bool decodingStarted = false;
    bool acceptingFrames = false;
    flicks acceptingFramesStartPTS{0};
    flicks playhead{0};

    /**
     * Set if ProcessOneSample encounter an error in the stream, or EOF
     */
    StopReason stopReason{StopReason::None};

public:
    using FrameCallback = std::function<void(AVFrame *, flicks, flicks)>;

    IDecoder()          = default;
    virtual ~IDecoder() = default;

    /**
     * Push one sample to the buffer
     * @param sample
     */
    void
    PushSample(const std::shared_ptr<PlayerCore::WrappedMediaSample> &sample)
    {
        this->sampleBuffer.PushSample(sample);
    }

    /**
     * Get duration of frames in sample buffer
     */
    flicks
    GetDurationInBuffer() const
    {
        return sampleBuffer.DurationInBuffer();
    }

    /**
     * Get the system timestamp when the last sample to be pushed to the buffer was received
     * Used to determine if the sample stream has stopped without notification
     */
    auto
    GetLastPushedSampleTime()
    {
        return this->sampleBuffer.LastPushedSample();
    }

    /**
     * Get the currect StopReason
     */
    StopReason
    GetStopReason()
    {
        return this->stopReason;
    }

    /**
     * The PTS of the last processed sample
     */
    flicks
    GetPlayhead() const
    {
        return this->playhead;
    }

    /**
     * Get the PTS where decoding started
     */
    flicks
    GetAcceptingFramesStartPTS() const
    {
        return this->acceptingFramesStartPTS;
    }

    /**
     * Set the PTS the stream should be fast forwarded to
     */
    void
    SetAcceptingFramesStartPTS(flicks pts)
    {
        this->acceptingFramesStartPTS = pts;
    }

    /**
     * Pop one sample from the sample buffer and push into the decoder
     * If the decoder has additional processing steps, such as scaling, this function also
     * forwards available frames from each stage to the next stage
     * @param idealPlayheadTimestamp The ideal time for the next frame to be decoded (used during fast-forward)
     * @return true if a sample was successfully popped and processed, if false it warrants a check if a StopReason is set
     */
    virtual bool ProcessOneSample(flicks idealPlayheadTimestamp) = 0;

    /**
     * Get one processed frame from the last stage of the decoder
     * @param cb A callback that gets called with the received frame
     * @return true if a frame was available, if false ProcessOneSample needs to be run
     */
    virtual bool GetOneFrame(const FrameCallback &cb) = 0;

    virtual void SetPlaybackRate(float rate) = 0;
};
}  // namespace Vape
