#include "./libav-decoder-audio.hpp"

#include "../processing/soundtouchwrapper.hpp"
#include "./libav-decoder-helpers.hpp"
#include "debug/log.hpp"

extern "C" {
#include <libavcodec/avcodec.h>
#include <libavfilter/avfilter.h>
}

namespace Vape {
const std::string_view h{"LibAVAudioDecoder"};

LibAVAudioDecoder::LibAVAudioDecoder(const IVideoOutputConsumer::AudioFormat &format)
    : decoderContext(nullptr)
    , filterGraph(nullptr)
    , audioFormat(format)
{
}

LibAVAudioDecoder::~LibAVAudioDecoder()
{
    avfilter_graph_free(&this->filterGraph);
    ReleaseDecoder(&this->decoderContext);
}

bool LibAVAudioDecoder::ProcessOneSample(flicks /*idealPlayheadTimestamp*/)
{
    if (auto packet = this->sampleBuffer.PopSample()) {
        if (packet->IsEOF()) {
            this->stopReason = StopReason::StreamEnded;
            return false;
        }

        //Log::TDebug(h, "PKT PTS: {}, {}", ToISO(packet->estimatedIngestTime), ToISO(Now()));
        this->averagePTStoIngestTimeOffset.Push(packet->GetPTSToIngestTimeOffset());

        // Check if we can drop this entire segment up to next sync sample
        if (!this->decodingStarted) {
            const flicks priorSyncSampleTime = toFlicks(0.2f);
            const flicks safeSyncDecodeTimestamp =
                this->acceptingFramesStartPTS - priorSyncSampleTime;
            auto pts = packet->PTS();

            if (packet->IsSyncSample() && (pts >= safeSyncDecodeTimestamp || pts > maxPTSSearch)) {
                this->decodingStarted = true;
            } else {
                return true;
            }
        }

        // Setup decoder if this is the first segment, or if this is a discontinuity
        // sample
        if (packet->IsDiscontinuitySample() || this->decoderContext == nullptr) {
            avfilter_graph_free(&this->filterGraph);
            ReleaseDecoder(&this->decoderContext);

            this->decoderContext = CreateAudioDecoder(h, *packet->audioConf);
            if (!this->decoderContext) {
                this->stopReason = StopReason::DecoderError;
                return false;
            }

            Log::TDebug(h, "Creating soundtouch wrapper");
            auto ctx         = this->decoderContext;
            this->soundTouch = std::make_unique<SoundTouchWrapper>(ctx->sample_rate, ctx->channels,
                                                                   ctx->channel_layout);
        }

        SendSampleToDecoder(this->decoderContext, *packet);
    } else {
        // Since neither the decoder or filter graph outputs frames without new
        // packets being input, we are clear return here
        return false;
    }

    GetFramesFromDecoder(this->decoderContext, INT_MAX, [&](AVFrame *frame, flicks pts) {
        if (!this->acceptingFrames) {
            if (pts >= this->acceptingFramesStartPTS /* || pts > maxPTSSearch */) {
                this->acceptingFrames = true;
                Log::TDebug(h, "Audio playback started at PTS {}", toFloat(pts));
            } else {
                return true;
            }
        }

        auto peaks = GetPeaksForFrame(frame);
        this->audioPeakSignal.Emit(peaks);

        this->soundTouch->SendAVFrame(frame);
        return true;
    });

    this->soundTouch->GetFrames(
        this->decoderContext->time_base, [&](AVFrame *frame, flicks /* pts */) {
            if (this->filterGraph == nullptr) {
                EnsureValidAVRational(&this->decoderContext->time_base,
                                      AVRational{1, this->decoderContext->sample_rate});

                this->filterGraph = CreateAudioFilterGraph(this->decoderContext, this->audioFormat);

                if (this->filterGraph == nullptr) {
                    this->stopReason = StopReason::DecoderError;
                    return false;
                }
            }

            SendFrameToFilterGraph(this->filterGraph, frame);
            return true;
        });

    return this->stopReason == StopReason::None;
}

bool
LibAVAudioDecoder::GetOneFrame(const IDecoder::FrameCallback &cb)
{
    if (!soundTouch) {
        return false;
    }

    int res = GetFramesFromFilterGraph(this->filterGraph, 1, [&](AVFrame *frame, flicks pts) {
        const flicks inputTimestamp = pts + this->soundTouch->GetRewriteOffset();
        const flicks estimatedIngestTimeForFrame =
            inputTimestamp + this->averagePTStoIngestTimeOffset.Average();

        cb(frame, pts, estimatedIngestTimeForFrame);

        return true;
    });

    return res > 0;
}

void
LibAVAudioDecoder::SetPlaybackRate(float rate)
{
    if (this->soundTouch) {
        this->soundTouch->SetTempo(rate);
    }
}

}  // namespace Vape
