#include "./libav-decoder-video.hpp"

#include "./libav-decoder-helpers.hpp"
#include "debug/log.hpp"

extern "C" {
#include <libavcodec/avcodec.h>
#include <libavfilter/avfilter.h>
}

namespace Vape {
const std::string_view h{"LibAVVideoDecoder"};

LibAVVideoDecoder::LibAVVideoDecoder(const IVideoOutputConsumer::VideoFormat &format)
    : decoderContext(nullptr)
    , filterGraph(nullptr)
    , ptsRewriter(format.frameDuration)
    , videoFormat(format)
{
}

LibAVVideoDecoder::~LibAVVideoDecoder()
{
    avfilter_graph_free(&this->filterGraph);
    ReleaseDecoder(&this->decoderContext);
}

bool
LibAVVideoDecoder::ProcessOneSample(flicks idealPlayheadTimestamp)
{
    bool success = true;

    if (auto packet = this->sampleBuffer.PopSample()) {
        if (packet->IsEOF()) {
            this->stopReason = StopReason::StreamEnded;
            return false;
        }

        // Log::Debug("PKT PTS: {}", packet->PTS.count());
        auto pts = packet->PTS();
        this->averagePTStoIngestTimeOffset.Push(packet->GetPTSToIngestTimeOffset());
        this->playhead = pts;

        // Check if we can drop this entire segment up to next sync sample
        if (!this->decodingStarted) {
            const flicks priorSyncSampleTime     = toFlicks(2.5f);
            const flicks safeSyncDecodeTimestamp = idealPlayheadTimestamp - priorSyncSampleTime;

            if (packet->IsSyncSample() &&
                (packet->estimatedIngestTime > safeSyncDecodeTimestamp || pts > maxPTSSearch)) {
                this->decodingStarted = true;
            } else {
                return false;
            }
        }

        // Setup decoder if this is the first segment, or if this is a discontinuity
        // sample
        if (packet->IsDiscontinuitySample() || this->decoderContext == nullptr) {
            avfilter_graph_free(&this->filterGraph);
            ReleaseDecoder(&this->decoderContext);

            this->decoderContext = CreateVideoDecoder(h, *packet->videoConf);
            if (!this->decoderContext) {
                this->stopReason = StopReason::DecoderError;
                return false;
            }
        }

        if (!SendSampleToDecoder(this->decoderContext, *packet)) {
            Log::TError(h, "Failed to send sample to video decoder");
            if (!IsHwDecoder(this->decoderContext)) {
                Log::TError(h, "Failed to send sample to software decoder, stopping playback");
                this->stopReason = StopReason::DecoderError;
                return false;
            }

            Log::TWarn(h, "Attempting to force software decoding");
            avfilter_graph_free(&this->filterGraph);
            ReleaseDecoder(&this->decoderContext);

            this->decoderContext = CreateVideoDecoder(h, *packet->videoConf, true);
            if (!this->decoderContext) {
                Log::TError(h, "Failed to create software decoder, stopping playback");
                this->stopReason = StopReason::DecoderError;
                return false;
            }

            if (!SendSampleToDecoder(this->decoderContext, *packet)) {
                Log::TError(h, "Failed to send sample to software decoder, stopping playback");
                this->stopReason = StopReason::DecoderError;
                return false;
            }
        }
    } else {
        // Since neither the decoder or filter graph outputs frames without new
        // packets being input, we are clear return here
        return false;
    }

    if (this->decoderContext) {
        bool timeBaseCorrected =
            EnsureValidAVRational(&this->decoderContext->time_base, AVRational{1, 90000});

        this->timeBaseCorrectedSignal.Emit(timeBaseCorrected);
    }

    GetFramesFromDecoder(this->decoderContext, INT_MAX, [&](AVFrame *frame, flicks pts) {
        if (this->filterGraph == nullptr) {
            auto ctx = this->decoderContext;

            if (!this->acceptingFrames) {
                auto estimatedIngestTimeForFrame =
                    this->averagePTStoIngestTimeOffset.Average() + pts;
                if (estimatedIngestTimeForFrame >= idealPlayheadTimestamp || pts > maxPTSSearch) {
                    this->acceptingFrames         = true;
                    this->acceptingFramesStartPTS = pts;
                    Log::TDebug(h, "Video playback started at PTS {}", toFloat(pts));
                } else {
                    return true;
                }
            }

            Log::TDebug(h, "Video time base set to {}:{}", ctx->time_base.den, ctx->time_base.num);

            auto &fmt = this->videoFormat;
            Log::TDebug(h, "Target frame duration is ", toFloat(fmt.frameDuration));

            this->filterGraph = CreateVideoFilterGraph(this->decoderContext, fmt);

            if (!this->filterGraph) {
                this->stopReason = StopReason::DecoderError;
                success          = false;
                return false;
            }

            return true;
        }

        CorrectAVFrameColorInfo(frame);

        SendFrameToFilterGraph(this->filterGraph, frame);

        return true;
    });

    return success;
}

bool
LibAVVideoDecoder::GetOneFrame(const IDecoder::FrameCallback &cb)
{
    int res = GetFramesFromFilterGraph(this->filterGraph, 1, [&](AVFrame *frame, flicks inputPTS) {
        // Log::Debug("OUT PTS: {}", pts.count());
        flicks duration{frame->pkt_duration};  // Duration is stored in flicks
        auto averageOffset = this->averagePTStoIngestTimeOffset.Average();

        this->ptsRewriter.ScheduleFrame(
            this->playbackRate, duration, INT_MAX, [&](flicks outputPTS) {
                const flicks estimatedIngestTime = inputPTS + averageOffset;
                cb(frame, outputPTS, estimatedIngestTime);
            });

        return true;
    });

    return res > 0;
}

void
LibAVVideoDecoder::SetPlaybackRate(float rate)
{
    this->playbackRate = rate;
}

}  // namespace Vape
