#include "./libav-decoder.hpp"

#include "../config.hpp"
#include "../consumer/ivideooutputconsumer.hpp"
#include "../settings/decoder.hpp"
#include "./avlogger.hpp"
#include "./libav-decoder-audio.hpp"
#include "./libav-decoder-helpers.hpp"
#include "./libav-decoder-video.hpp"
#include "debug/log.hpp"
#include "fundamentals/deferred-execution.hpp"
#include "fundamentals/profiling-timer.hpp"

extern "C" {
#include <libavformat/avformat.h>
#include <libavutil/version.h>
}

#include <iostream>
#include <random>
#include <utility>

using namespace twitch;

namespace Vape {

static std::string_view h{"LibAVDecoder"};

inline IDecoder &
LibAVDecoder::GetDecoder(StreamIndex index)
{
    assert(index == StreamIndex::Audio || index == StreamIndex::Video);
    return *this->decoders[static_cast<int>(index)];
}

inline PlaybackRateMonitor &
LibAVDecoder::GetRateMonitor(StreamIndex index)
{
    assert(index == StreamIndex::Audio || index == StreamIndex::Video);
    return this->rateMonitors[static_cast<int>(index)];
}

LibAVDecoder::LibAVDecoder(IVideoOutputConsumer *_consumer, flicks delay)
    : consumer(_consumer)
    , decoders({
          std::make_unique<LibAVAudioDecoder>(consumer->GetAudioFormat()),
          std::make_unique<LibAVVideoDecoder>(consumer->GetVideoFormat()),
      })
{
#if LIBAVCODEC_VERSION_INT < AV_VERSION_INT(58, 9, 100)
    av_register_all();
#endif

    av_log_set_callback(AVLogCallback);

    {
        auto &rateMonitor = this->GetRateMonitor(StreamIndex::Video);
        rateMonitor.SetOffsetMultiplier(3.0f);
        rateMonitor.SetOffsetClamp(-0.05f, 0.05f);
        rateMonitor.SetAggressiveOffsetClamp(-0.5f, 0.5f);
        rateMonitor.SetMinOffset(0.01f);
    }

    {
        auto &rateMonitor = this->GetRateMonitor(StreamIndex::Audio);
        rateMonitor.SetOffsetMultiplier(3.0f);
        rateMonitor.SetOffsetClamp(-0.05f, 0.05f);
        rateMonitor.SetAggressiveOffsetClamp(-0.5f, 0.5f);
        rateMonitor.SetMinOffset(0.01f);
    }

    this->offsetTracker.SetDesiredDelay(delay);
}

LibAVDecoder::~LibAVDecoder()
{
}

bool
LibAVDecoder::DecoderTick(StreamIndex index, flicks targetInsertTime)
{
    // Start a timer that signal it's duration when the executor goes out of scope
    ProfilingTimer profilingTimer;
    DeferredExecution deferredProfilingTimerReport([&]() {
        this->decodeProcessingTimeSignal.Emit(
            index, {profilingTimer.GetStartTime(), profilingTimer.GetTime()});
    });

    // Get the ideal timestamp for the playhead
    auto idealPlayheadTimestamp = this->offsetTracker.GetIdealPlayheadTimestamp();
    auto &decoder               = this->GetDecoder(index);

    // Signal the current length of the sample buffer
    this->sampleBufferLengthSignal.Emit(index, decoder.GetDurationInBuffer());

    // Callback for GetOneFrame
    auto handleFrame = [&](AVFrame *frame, flicks /*pts*/, flicks timestamp) {
        // Calculate how much we deviate from the target offset
        const float offset = toFloat(timestamp - targetInsertTime);
        this->offsetSignal.Emit(index, offset);

        // Get the rate monitor for this stream and calculate new playback rate
        auto &rateMonitor        = this->GetRateMonitor(index);
        const float playbackRate = rateMonitor.CalculatePlaybackRateFromOffset(offset);

        // Set new playback rate to decoder
        decoder.SetPlaybackRate(playbackRate);

        // Signal playback rate
        this->speedSignal.Emit(index, playbackRate);

        // Schedule frame to output device
        this->consumer->ScheduleFrame(index, frame, timestamp);
    };

    // Try to get one frame from the decoder
    if (decoder.GetOneFrame(handleFrame)) {
        return true;
    }

    // Make the decoder process one more sample to be ready on next tick
    if (!decoder.ProcessOneSample(idealPlayheadTimestamp)) {
        // If this is the video decoder, signal the PST that will be used in player-core
        if (index == StreamIndex::Video) {
            this->playheadSignal.Emit(decoder.GetPlayhead());
        }

        if (auto reason = decoder.GetStopReason(); reason != StopReason::None) {
            this->stopSignal.Emit(reason);
            return false;
        }
    }

    return true;
}

void
LibAVDecoder::PushStreamData(StreamIndex index,
                             const std::shared_ptr<PlayerCore::WrappedMediaSample> &sample)
{
    // Get the decoder by index
    auto &decoder = this->GetDecoder(index);
    decoder.PushSample(sample);
}

void
LibAVDecoder::Run()
{
    bool consumerStarted        = this->consumer->Started();
    auto idealPlayheadTimestamp = this->offsetTracker.GetIdealPlayheadTimestamp();

    // Test if the consumer needs more frames from the decoder, and run one decoder tick if needed
    auto handleDecoder = [&](StreamIndex index) -> bool {
        // Test if video buffer needs more frames
        flicks bufferLength       = this->consumer->BufferLength(index);
        flicks targetBufferLength = this->consumer->TargetBufferLength(index);
        bool canBeStarted         = true;

        if (bufferLength < targetBufferLength) {
            // Set that video can't be started since we want to start playback on a
            // clean iteration of Run()
            canBeStarted = false;

            flicks idealInsertTimestamp = idealPlayheadTimestamp;
            if (consumerStarted) {
                idealInsertTimestamp += bufferLength;
            } else {
                idealInsertTimestamp += targetBufferLength;
            }

            this->DecoderTick(index, idealInsertTimestamp);
        }

        this->decodedBufferLengthSignal.Emit(index, bufferLength, targetBufferLength);

        return canBeStarted;
    };

    // Maybe run decoder for video
    bool videoCanBeStarted = handleDecoder(StreamIndex::Video);

    // If video is ready to start, transfer the start PTS to the audio decoder to fast forward to
    if (videoCanBeStarted && !this->ptsStartTransfered) {
        auto &vdec = this->GetDecoder(StreamIndex::Video);
        auto &adec = this->GetDecoder(StreamIndex::Audio);

        auto pts = vdec.GetAcceptingFramesStartPTS();
        adec.SetAcceptingFramesStartPTS(pts);

        this->ptsStartTransfered = true;
        Log::TInfo(h, "Transfered start PTS from video decoder to audio decoder: {}", toFloat(pts));

        this->ptsSearchDoneSignal.Emit();
    }

    bool audioCanBeStarted = false;
    if (this->ptsStartTransfered) {
        // Maybe run the decoder for audio
        audioCanBeStarted = handleDecoder(StreamIndex::Audio);
    }

    if (!consumerStarted && videoCanBeStarted && audioCanBeStarted) {
        for (auto &rateMonitor : this->rateMonitors) {
            rateMonitor.SetToStarted();
        }

        this->consumer->Start();
        this->startedSignal.Emit();
    }

    // Check if it was more than 30s since we got the last MediaSample from
    // player-core and restart playback
    const auto now          = system_clock::now();
    const auto minTimePoint = now - std::chrono::seconds{30};
    if (this->GetDecoder(StreamIndex::Video).GetLastPushedSampleTime() < minTimePoint) {
        Log::Error("Playback run loop stopped because it's more than 30 seconds "
                   "since we recieved the last video sample from player-core.");
        this->stopSignal.Emit(StopReason::SampleStreamStopped);
    } else if (this->GetDecoder(StreamIndex::Audio).GetLastPushedSampleTime() < minTimePoint) {
        Log::Error("Playback run loop stopped because it's more than 30 seconds "
                   "since we recieved the last audio sample from player-core.");
        this->stopSignal.Emit(StopReason::SampleStreamStopped);
    }
}

void
LibAVDecoder::SetMinTotalDelay(flicks newMinTotalDelay)
{
    this->offsetTracker.SetMinTotalDelay(newMinTotalDelay);
    this->totalDelaySignal.Emit(this->offsetTracker.GetTotalDelay());
}

void
LibAVDecoder::UpdateSettings(const DecoderSettings &s)
{
    this->offsetTracker.SetAdditionalDelay(toFlicks(s.channelOffset));
    this->totalDelaySignal.Emit(this->offsetTracker.GetTotalDelay());
}

void
LibAVDecoder::StartAggressiveRecovery()
{
    for (auto &rateMonitor : this->rateMonitors) {
        rateMonitor.UseAggressive();
    }
}

Signal::Provider<bool> &
LibAVDecoder::timeBaseCorrectedSignal()
{
    auto &decoder      = this->GetDecoder(StreamIndex::Video);
    auto &videoDecoder = static_cast<LibAVVideoDecoder &>(decoder);
    return videoDecoder.timeBaseCorrectedSignal;
}

Signal::Provider<std::vector<float>> &
LibAVDecoder::audioPeakSignal()
{
    auto &decoder      = this->GetDecoder(StreamIndex::Audio);
    auto &audioDecoder = static_cast<LibAVAudioDecoder &>(decoder);
    return audioDecoder.audioPeakSignal;
}

}  // namespace Vape
