#pragma once

#include "../processing/playbackrate.hpp"
#include "../status/perfmon.hpp"
#include "../status/statusindicator.hpp"
#include "../stop-reason.hpp"
#include "fundamentals/locked-value.hpp"
#include "fundamentals/signal-provider.hpp"
#include "vapour-player-core/time.hpp"
#include "vapour-player-core/types/streamindex.hpp"
#include "vapour-player-core/types/wrappedmediasample.hpp"

#include <array>
#include <memory>
#include <mutex>
#include <queue>
#include <vector>

namespace Vape {

class IVideoOutputConsumer;
class IDecoder;
struct DecoderSettings;

class LibAVDecoder
{
    IVideoOutputConsumer *const consumer;

    std::array<std::unique_ptr<IDecoder>, static_cast<int>(StreamIndex::AVStreamCount)> decoders;
    std::array<PlaybackRateMonitor, static_cast<int>(StreamIndex::AVStreamCount)> rateMonitors;

    IDecoder &GetDecoder(StreamIndex index);
    PlaybackRateMonitor &GetRateMonitor(StreamIndex index);

    OffsetTracker offsetTracker;
    bool ptsStartTransfered{false};

    bool DecoderTick(StreamIndex index, flicks targetInsertTime);

public:
    LibAVDecoder(IVideoOutputConsumer *consumer, flicks delay);
    ~LibAVDecoder();

    void PushStreamData(StreamIndex index,
                        const std::shared_ptr<PlayerCore::WrappedMediaSample> &sample);
    void Run();
    void SetMinTotalDelay(flicks newMinTotalDelay);

    void UpdateSettings(const DecoderSettings &);
    void StartAggressiveRecovery();

    /**
     * Time spent decoding and converting a single frame
     * @param1 Time spent in Consume{Stream}Packet
     */
    Signal::Provider<StreamIndex, PerfMon::Record> decodeProcessingTimeSignal;

    // The length of the sample buffer
    Signal::Provider<StreamIndex, flicks> sampleBufferLengthSignal;

    /**
     * The length of the decoded frames buffer
     * @param1 Current size
     * @param2 Target size
     */
    Signal::Provider<StreamIndex, flicks, flicks> decodedBufferLengthSignal;
    Signal::Provider<> ptsSearchDoneSignal;
    Signal::Provider<> startedSignal;
    Signal::Provider<StopReason> stopSignal;
    Signal::Provider<bool> &timeBaseCorrectedSignal();
    Signal::Provider<StreamIndex, float> offsetSignal;
    Signal::Provider<StreamIndex, float> speedSignal;
    Signal::Provider<std::vector<float>> &audioPeakSignal();
    Signal::Provider<flicks> playheadSignal;
    Signal::Provider<flicks> totalDelaySignal;
};

}  // namespace Vape
