#pragma once
#include <cstdint>
#include <vector>

namespace Vape {
/**
 * Represents an RGBA pixel in an image
 */
struct Pixel {
    /**
     * The components of the pixel
     */
    uint8_t r, g, b, a;

    /**
     * Construct a pixel from components
     */
    constexpr Pixel(uint8_t _r, uint8_t _g, uint8_t _b, uint8_t _a);

    /**
     * Construct a pixel from packed ARGB
     */
    static constexpr Pixel FromARGB(uint32_t p);

    /**
     * Construct a pixel from packed RGBX
     */
    static constexpr Pixel FromRGBA(uint32_t p);

    /**
     * Construct a pixel from packed BGRA
     */
    static constexpr Pixel FromBGRA(uint32_t p);

    /**
     * Construct a pixel from monochrome components
     * Y gets copied to R, G and B
     */
    static constexpr Pixel FromMonochrome(uint8_t y, uint8_t a);

    /**
     * Overlay this pixel with another pixel
     */
    void Overlay(const Pixel &in);

    /**
     * Export ARGB packed pixel
     */
    constexpr uint32_t ToARGB();

    /**
     * Export RGBX packed pixel
     */
    constexpr uint32_t ToRGBA();

    /**
     * Export BGRA packed pixel
     */
    constexpr uint32_t ToBGRA();
};

/**
 * Represents an image and it's margins
 */
struct Image {
    int width{0}, height{0};
    std::vector<Pixel> data;

    /**
     *  Margins that should be applied to conform with the brand guidelines
     *  These are allowed to be negative to indicate the image has spill (shadow, etc)
     */
    struct Margins {
        int top{0}, right{0}, bottom{0}, left{0};
    } margin;

    /**
     * Get a pixel of the image from coordinates
     * This function does not do bounds checks
     */
    const Pixel &GetPixel(size_t x, size_t y) const;
};

constexpr Pixel::Pixel(uint8_t _r, uint8_t _g, uint8_t _b, uint8_t _a)
    : r(_r)
    , g(_g)
    , b(_b)
    , a(_a)
{
}

constexpr Pixel
Pixel::FromARGB(uint32_t p)
{
    uint8_t a = p & 0xFF;
    uint8_t r = (p >> 8) & 0xFF;
    uint8_t g = (p >> 16) & 0xFF;
    uint8_t b = (p >> 24) & 0xFF;

    return {r, g, b, a};
}

constexpr Pixel
Pixel::FromRGBA(uint32_t p)
{
    uint8_t r = p & 0xFF;
    uint8_t g = (p >> 8) & 0xFF;
    uint8_t b = (p >> 16) & 0xFF;
    uint8_t a = (p >> 24) & 0xFF;

    return {r, g, b, a};
}

constexpr Pixel
Pixel::FromBGRA(uint32_t p)
{
    uint8_t b = p & 0xFF;
    uint8_t g = (p >> 8) & 0xFF;
    uint8_t r = (p >> 16) & 0xFF;
    uint8_t a = (p >> 24) & 0xFF;

    return {r, g, b, a};
}

constexpr Pixel
Pixel::FromMonochrome(uint8_t y, uint8_t a)
{
    return {y, y, y, a};
}

inline void
Pixel::Overlay(const Pixel &in)
{
    // NOTE(Love): This code makes the assumption that the target has full alpha and will
    // not give the correct result for overlaying pixels on a semi transparent target
    // It also ignores gamma, if we want to use this for more than just the logo, it needs some more work
    r = ((uint32_t)r * (255 - in.a) + (uint32_t)in.r * in.a) / 255;
    g = ((uint32_t)g * (255 - in.a) + (uint32_t)in.g * in.a) / 255;
    b = ((uint32_t)b * (255 - in.a) + (uint32_t)in.b * in.a) / 255;
    a = 0xFF;
}

constexpr uint32_t
Pixel::ToARGB()
{
    return (b << 24) | (g << 16) | (r << 8) | a;
}

constexpr uint32_t
Pixel::ToRGBA()
{
    return (a << 24) | (b << 16) | (g << 8) | r;
}

constexpr uint32_t
Pixel::ToBGRA()
{
    return (a << 24) | (r << 16) | (g << 8) | b;
}

inline const Pixel &
Image::GetPixel(size_t x, size_t y) const
{
    return this->data[this->width * y + x];
}

}  // namespace Vape
