#include "./placeholder-frame.hpp"

#include "./twitch-logo.hpp"

#include <array>

namespace Vape {

static void
CreateBlackPlaceholderFrameBGRA(uint32_t *data, int width, int height)
{
    constexpr uint32_t color = Pixel(0, 0, 0, 0xff).ToBGRA();
    std::fill(data, data + (width * height), color);
}

static void
CreateGlitchPlaceholderFrameBGRA(uint32_t *data, int width, int height)
{
    // Fill entire frame with purple
    constexpr uint32_t color = GetTwitchPurple().ToBGRA();
    std::fill(data, data + (width * height), color);

    Image &glitch = GetTwitchLogo();

    // Caluculate position from top left corner
    int imgW    = glitch.width;
    int imgH    = glitch.height;
    int offsetX = width - imgW - glitch.margin.right;
    int offsetY = height - imgH - glitch.margin.bottom;

    // Iterate rows
    for (int y = 0; y < imgH; y++) {
        // Output pixel y position
        int oy = y + offsetY;

        if (oy < 0 || oy >= height) {
            continue;
        }

        // Iterate columns
        for (int x = 0; x < imgW; x++) {
            // Output pixel x position
            int ox = x + offsetX;

            if (ox < 0 || ox >= width) {
                continue;
            }

            // Get reference to output pixel
            uint32_t &out = data[oy * width + ox];

            // Convert packed output pixel to Pixel struct
            auto outPixel = Pixel::FromBGRA(out);

            // Overlay glitch pixel
            outPixel.Overlay(glitch.GetPixel(x, y));

            // Write result to output
            out = outPixel.ToBGRA();
        }
    }
}

static void
CreateBarsPlaceholderFrameBGRA(uint32_t *data, int width, int height, uint8_t max)
{
    const std::array<uint32_t, 8> colors = {
        Pixel(max, max, max, 0xFF).ToBGRA(), Pixel(max, max, 0, 0xFF).ToBGRA(),
        Pixel(0, max, max, 0xFF).ToBGRA(),   Pixel(0, max, 0, 0xFF).ToBGRA(),
        Pixel(max, 0, max, 0xFF).ToBGRA(),   Pixel(max, 0, 0, 0xFF).ToBGRA(),
        Pixel(0, 0, max, 0xFF).ToBGRA(),     Pixel(0, 0, 0, 0xFF).ToBGRA(),
    };

    // The fast paths look identical, but gets separately unrolled
    if (width == 1920) {
        const int columnW = width / colors.size();

        for (int y = 0; y < height; ++y) {
            for (auto col : colors) {
                std::fill(data, data + columnW, col);
                data += columnW;
            }
        }
    } else if (width == 1280) {
        const int columnW = width / colors.size();

        for (int y = 0; y < height; ++y) {
            for (auto col : colors) {
                std::fill(data, data + columnW, col);
                data += columnW;
            }
        }
    } else {
        // Slow path for SD resolutions or if we introduce custom resolutions
        const int columnW = width / colors.size();

        for (int y = 0; y < height; ++y) {
            for (int x = 0; x < width; ++x) {
                *data++ = colors[x / columnW];
            }
        }
    }
}

void
CreatePlaceholderFrameBGRA(uint32_t *data, int width, int height, PlaceholderType type)
{
    switch (type) {
        case PlaceholderType::Glitch:
            CreateGlitchPlaceholderFrameBGRA(data, width, height);
            break;
        case PlaceholderType::Bars100:
            CreateBarsPlaceholderFrameBGRA(data, width, height, 255);
            break;
        case PlaceholderType::Bars75:
            CreateBarsPlaceholderFrameBGRA(data, width, height, 191);
            break;
        case PlaceholderType::Black:
        default:
            CreateBlackPlaceholderFrameBGRA(data, width, height);
            break;
    }
}

}  // namespace Vape
