#include "./player.hpp"

#include "../config.hpp"
#include "../consumer/consumer-factory.hpp"
#include "../consumer/decklink-base.hpp"
#include "../consumer/decklink-output.hpp"
#include "../consumer/ndi-output.hpp"
#include "../credentials/manager.hpp"
#include "../decoder/libav-decoder.hpp"
#include "../settings/manager.hpp"
#include "../status/manager.hpp"
#include "../utils/thread-affinity.hpp"
#include "api/client.hpp"
#include "debug/log.hpp"
#include "fundamentals/deferred-execution.hpp"
#include "fundamentals/rapidjson.hpp"
#include "fundamentals/string-helpers.hpp"

namespace Vape {

using namespace std::chrono_literals;

constexpr auto decoderLoopDelay = 1ms;

std::shared_ptr<LibAVDecoder>
CreateDecoder(IVideoOutputConsumer *consumer, flicks desiredDelay)
{
    return std::make_shared<LibAVDecoder>(consumer, desiredDelay);
}

Player::Player(SettingsManager &settingsManager, Signal::Group &signalGroup,
               Signal::ListenerPtrList &signalList, API::Client &apiClient)
    : h("Player")
    , outputType(GetOutputType(Config::Get().outputType))
    , fetcher(apiClient)
    , consumerManager(this->outputType)
{
    switch (this->outputType) {
        case OutputType::NDI: {
            const auto &c = Config::Get();
            if (c.ndiOutputName.empty()) {
                throw std::runtime_error("Missing NDI output name environment variable "
                                         "(VAPOUR_MULTISTREAM_NDI_OUTPUT_NAME)");
            }
        } break;

        case OutputType::DeckLink: {
            // Initiate DeckLinkScanner instance and signals
            auto &scanner = DeckLinkScanner::instance();
            signalList.push_back(scanner.deviceNamesSignal.AddIndirectListener(
                signalGroup, bind_this(&settingsManager, &SettingsManager::SetDeviceNames)));
            scanner.Init();
        } break;

        default:
            break;
    }
}

Player::~Player()
{
    switch (this->outputType) {
        case OutputType::DeckLink: {
            // Deinit DeckLinkScanner, will uninstall the callback and release all cached outputs
            DeckLinkScanner::instance().Deinit();
        } break;

        default:
            break;
    }
}

StopReason
Player::Run(SettingsManager &settingsManager, StatusManager &statusManager,
            CredentialsManager &credentialsManager, StartLock &startLock)
{
    auto signalInitializedImpl = [&startLock]() {
        // Signal to the caller that the initalization is done, or that we have errored out
        {
            std::unique_lock lock(startLock.mutex);
            startLock.initialized = true;
        }
        startLock.cv.notify_one();
    };

    // unique_ptr that will signal caller onece reset, or when we error out and unique_ptr gets freed automatically
    auto signalInitialized =
        std::make_unique<DeferredExecution<decltype(signalInitializedImpl)>>(signalInitializedImpl);

    // The state should be safe to read here, as this is a new thread, and the execution of
    // the main thread is paused until the worker token is destructred.
    auto state = settingsManager.state;
    auto &p    = state.player;

    // Signal listener list and signal group, local to this thread and this run of the player
    Signal::ListenerPtrList listeners;
    Signal::Group signalGroup;

    // Helper function to subscribe to signals using the local signal group
    auto subscribe = [&listeners, &signalGroup](auto &signal, auto fn) {
        listeners.push_back(signal.AddIndirectListener(signalGroup, fn));
    };

    auto req      = this->consumerManager.GetConsumer(state.output);
    auto consumer = req.first;
    auto result   = req.second;

    if (result <= ConsumerManager::GetResult::None || !consumer) {
        return StopReason::ConsumerError;
    }

    // Validate that these settings can even be played. This is done after PrepareConsumer so mode options are up-to-date
    if (!p.enabled) {
        return StopReason::OutputNotEnabled;
    } else if (p.streamName.empty()) {
        return StopReason::StreamNameEmpty;
    }

    // If this is an DeckLink consumer, set up the extra data links
    if (auto deckLink = dynamic_cast<DeckLinkOutput *>(consumer)) {
        Log::TTrace(h, "Setting up DeckLink signals");

        // Set captured settings
        deckLink->UpdateSettings(state.deckLink);

        // Subscribe to updated settings for DeckLink
        subscribe(settingsManager.deckLinkSettingsSignal,
                  bind_this(deckLink, &DeckLinkOutput::UpdateSettings));

        // Set up StatusManager DeckLink signals
        statusManager.SubscribeToDeckLink(listeners, *deckLink);
    }

    // Initialize consumer
    if (!consumer->Valid()) {
        Log::TError(this->h, "Consumer is not valid");
        return StopReason::ConsumerError;
    }

    // Create decoder
    auto decoder = CreateDecoder(consumer, p.desiredDelay);
    if (!decoder) {
        Log::TError(this->h, "Failed to create decoder");
        return StopReason::DecoderError;
    }

    // Set up StatusManager decoder signals
    statusManager.SubscribeToDecoder(listeners, *decoder);
    statusManager.SubscribeToTwitchStreamFetcher(listeners, this->fetcher);

    // Set captured decoder settings
    decoder->UpdateSettings(state.decoder);

    // Subscibe to updated settings
    subscribe(settingsManager.decoderSettingsSignal,
              bind_this(decoder.get(), &LibAVDecoder::UpdateSettings));
    subscribe(settingsManager.aggressiveRecoverySignal,
              bind_this(decoder.get(), &LibAVDecoder::StartAggressiveRecovery));

    // Set up link between sink and decoder
    // ... media samples from sink
    subscribe(this->fetcher.sampleSignal, [&decoder](auto index, auto &sample) {
        switch (index) {
            case StreamIndex::Video:
            case StreamIndex::Audio:
                decoder->PushStreamData(index, sample);
                break;

            default:
                break;
        }
    });

    // Subscribe to decoder playhead to update sink
    subscribe(decoder->playheadSignal,
              bind_this(&this->fetcher, &PlayerCore::Listener::updatePlayhead));

    // The default stop reason that gets updated by stop signals from components
    auto stopReason = StopReason::None;

    // Callback for component stop signals
    auto componentStop = [&q = this->quitter, &stopReason](auto s) {
        // Update stop reason
        stopReason = s;

        // Set quitter to exit loop
        q.Quit();
    };

    // Subscribe to stop signals from...
    subscribe(decoder->stopSignal, componentStop);   // LibAVDecoder
    subscribe(consumer->stopSignal, componentStop);  // Consumer

    // Reset the deferred executor that signals initialization is done
    signalInitialized.reset();

    // Configure TwitchStreamFetcher, this is done after the sync point as to not lock up the main thread
    auto [clientID, accessToken] = credentialsManager.ResolveCredentials();
    //TODO: Test why not working
    this->fetcher.setCredentials(clientID, accessToken);
    Log::Debug("{} {}", clientID, accessToken);
    this->fetcher.setTranscodeLimits(p.maxTranscode);

    subscribe(this->fetcher.minTotalDelaySignal,
              bind_this(decoder.get(), &LibAVDecoder::SetMinTotalDelay));

    /**
     * Configure thread
     */
    SetThreadNameAndAffinity("VMS " + h);

    Log::TDebug(this->h, "Play {}", p.streamName);
    std::string url;

    // Expand the stream name to an URL
    if (StartsWith(p.streamName, "lvs:")) {
        // Handle stream names with lvs: prefix
        auto split = Split(p.streamName, ":", 2);
        if (split.size() == 3 && !split[1].empty() && !split[2].empty()) {
            url = fS("https://usher.ttvnw.net/api/lvs/hls/lvs.{}.{}.m3u8?allow_source=true",
                     split[1], split[2]);
        }
    } else if (StartsWith(p.streamName, "http://") || StartsWith(p.streamName, "https://")) {
        // Handle stream names that are URLs by using them as is
        url = p.streamName;
    } else {
        // Handle default case
        url = fS("https://twitch.tv/{}", p.streamName);
    }

    // Start stream fetcher
    this->fetcher.play(url, p.desiredDelay);

    // Signal player start
    this->startedSignal.Emit();

    // Run loop with wait until quitter is set to stop
    while (!this->quitter.WaitFor(decoderLoopDelay)) {
        // Poll signals
        signalGroup.Poll();

        // Run one decoder tick
        decoder->Run();
    }

    Log::TDebug(h, "Exited decode loop");

    // Stop output device and fetcher
    consumer->Stop();

    Log::TDebug(h, "Stopped consumer");

    this->fetcher.pause();

    Log::TDebug(h, "Paused player-core");

    this->stopSignal.Emit(stopReason);

    return stopReason;
}

}  // namespace Vape
